# @package      hubzero-submit-client
# @file         ClientIdAuthAttributes.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import subprocess
import getpass
import pwd
import logging

class ClientIdAuthAttributes:
   def __init__(self):
      self.logger            = logging.getLogger(__name__)
      self.sessionAttributes = {}
      self.userAttributes    = {}
      self.ppkAttributes     = {}

      if 'SESSIONDIR' in os.environ:
         resourcePath = os.path.join(os.environ['SESSIONDIR'],'resources')
         self.getSessionAttributes(resourcePath)

      if not 'sessionToken' in self.sessionAttributes:
         if 'HOME' in os.environ:
            resourcePath = os.path.join(os.environ['HOME'],'.default_resources')
            self.getSessionAttributes(resourcePath)


   def getSessionAttributes(self,
                            resourcePath):
      del self.sessionAttributes
      self.sessionAttributes = {}

      if os.path.exists(resourcePath):
         try:
            fpResource = open(resourcePath,'rb')
            try:
               eof = False
               while not eof:
                  encodedRecord = fpResource.readline()
                  try:
                     record = encodedRecord.decode('utf-8')
                  except UnicodeDecodeError:
                     pass
                  else:
                     if record != "":
                        if   record.startswith('session_token '):
                           self.sessionAttributes['sessionToken'] = record.split()[1]
                        elif record.startswith('sessionid '):
                           self.sessionAttributes['sessionId'] = record.split()[1]
                        elif record.startswith('cache_hosts '):
                           self.sessionAttributes['cacheHosts'] = record.split()[1]
                     else:
                        eof = True
            except:
               pass
            finally:
               fpResource.close()
         except (IOError,OSError):
            pass


   def getUserAttributes(self):
      del self.ppkAttributes
      self.ppkAttributes = {}
      if 'SUBMITPRIVATEKEYPATH' in os.environ:
         self.getPPKAttributes()

      del self.userAttributes
      self.userAttributes = {}
      self.userAttributes['userName'] = getpass.getuser()
      if 'SUDO_USER' in os.environ:
         self.userAttributes['sudoUserName'] = os.environ['SUDO_USER']
      if 'WS_USER' in os.environ:
         self.userAttributes['wsUserName'] = os.environ['WS_USER']

      havePPK = 'privateFingerPrint' in self.ppkAttributes
      haveToken = 'sessionToken' in self.sessionAttributes
      if not haveToken and not havePPK:
         si = sys.stdin
         try:
            sys.stdin = open('/dev/tty')
            try:
               try:
                  username = raw_input("Username: (%s) " % (self.userAttributes['userName']))
               except NameError:
                  try:
                     username = input("Username: (%s) " % (self.userAttributes['userName']))
                  except (IOError,OSError):
                     username = None
               except (IOError,OSError):
                  username = None

               if username:
                  self.userAttributes['userName'] = username
               password = getpass.getpass("Password: ")
               if password != "":
                  self.userAttributes['password'] = password
            except (IOError,OSError):
               pass
         except (IOError,OSError):
            pass
         sys.stdin = si

      try:
         (login,pw,uid,gid,name,homedir,shell) = pwd.getpwnam(self.userAttributes['userName'])
         self.userAttributes['userId'] = uid
      except:
         pass


   def getPPKAttributes(self):
      try:
         submitPrivateKeyPath = os.environ['SUBMITPRIVATEKEYPATH']
      except:
         pass
      else:
         if os.path.exists(submitPrivateKeyPath):
            command = ['openssl','rsa','-in',submitPrivateKeyPath,'-noout','-modulus']
            child = subprocess.Popen(command,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            stdOutput,stdError = child.communicate()
            exitStatus = child.returncode
            if exitStatus == 0:
               modulus = stdOutput.split(b'=')[1].strip()
               command = ['openssl','dgst','-sha256']
               child = subprocess.Popen(command,
                                        stdin=subprocess.PIPE,
                                        stdout=subprocess.PIPE,
                                        stderr=subprocess.PIPE,
                                        close_fds=True)
               stdOutput,stdError = child.communicate(modulus)
               exitStatus = child.returncode
               if exitStatus == 0:
                  privateFingerPrint = stdOutput.split(b' ')[1].strip()
                  if isinstance(privateFingerPrint,bytes):
                     privateFingerPrint = privateFingerPrint.decode('utf-8')
                  self.ppkAttributes['privateFingerPrint'] = privateFingerPrint


   def haveIdAuthAttributes(self):
      haveAttributes = False
      if   'sessionToken' in self.sessionAttributes:
         haveAttributes = True
      elif 'userName' in self.userAttributes and 'password' in self.userAttributes:
         haveAttributes = True
      elif 'userName' in self.userAttributes and 'privateFingerPrint' in self.ppkAttributes:
         haveAttributes = True

      return(haveAttributes)


   def getSignonAttributes(self):
      signonAttributes = {}
      signonAttributes['userName'] = self.userAttributes['userName']

      userAttributes = ['sudoUserName','wsUserName','password','userId']
      for userAttribute in userAttributes:
         try:
            signonAttributes[userAttribute] = self.userAttributes[userAttribute]
         except:
            pass

      sessionAttributes = ['sessionId','sessionToken','cacheHosts']
      for sessionAttribute in sessionAttributes:
         try:
            signonAttributes[sessionAttribute] = self.sessionAttributes[sessionAttribute]
         except:
            pass

      ppkAttributes = ['privateFingerPrint']
      for ppkAttribute in ppkAttributes:
         try:
            signonAttributes[ppkAttribute] = self.ppkAttributes[ppkAttribute]
         except:
            pass

      return(signonAttributes)


   def clearSessionToken(self):
      if 'sessionToken' in self.sessionAttributes:
         del self.sessionAttributes['sessionToken']


