#!/usr/bin/env python3
#
# @package      hubzero-submit-distributor
# @file         distributor.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
"""Distribute computation jobs to remote resources

   Generate a pair of script files for remote job submission.
   The first file is the batch submission script, the second is a shell script
   that wraps the application and generates time usage data.

   RUN AS FOLLOWS:
     distributor --jobid <jobId> --venue <venue> --inputfile <inputfile>
                 --nCpus <nCpus> --ppn <ppn> --nGpus <nGpus> --gpn <gpn>
                 --memory <MB> --wallTime <wallTime>
                 --env <environment variable> --manager <manager>

   <jobId>                 is the job identifier
   <executable>            is the MPI enabled executable to be run.
   <inputfile>             is a file to be sent to the remote site
   <outputfile>            is a file to be retrieved from the remote site
   <nCpus>                 the number of processors to use
   <ppn>                   the processors/node to use
   <nGpus>                 the number of GPUs to use
   <gpn>                   the GPUs/node to use
   <MB>                    memory requirement MB/node
   <wallTime>              wall time limit for remote process
   <environment variable>  variable=value
   <venue>                 remote venue
   <manager>               multi-processor control manager

                                                  | cpu time    |
                                                  | from time() |
                                                  |             |
                                      |  waiting  | real(wall)  |
                                      |   time    |    time     |
                                      |           |             |
     |------------|---------|---------|-----------|-------------|----------|----
  simulate     condor     site      files     simulation    simulation   files
   button        job    selected    staged     started        ended      staged
  pressed    submitted               in                                   out

   """
import os
import sys
import logging

from hubzero.submit.ApplicationUtilities import ApplicationUtilities as ApplicationUtilities
from hubzero.submit.JobDistributor       import JobDistributor

DISTRIBUTORLOGLEVEL    = "INFO"
DISTRIBUTORSYSLOGLOCAL = "LOCAL3"

#DISTRIBUTORLOGDIRECTORY = os.path.join(os.sep,'var','log','submit','distributor')
CONFIGURATIONDIRECTORY       = os.path.join(os.sep,'etc','submit')
DISTRIBUTORCONFIGURATIONFILE = 'distributor.conf'
DAEMONSDIRECTORY             = os.path.join(os.sep,"etc","submit")
DAEMONSCONFIGURATIONFILE     = 'daemons.conf'
INFOSDIRECTORY               = os.path.join(os.sep,"etc","submit")
INFOSCONFIGURATIONFILE       = 'infos.conf'

def distributor(configurationDirectory,
                distributorConfigurationFile,
                daemonsDirectory,
                daemonsConfigurationFile,
                infosDirectory,
                infosConfigurationFile):
   jobDistributor = JobDistributor(configurationDirectory,
                                   distributorConfigurationFile,
                                   daemonsDirectory,
                                   daemonsConfigurationFile,
                                   infosDirectory,
                                   infosConfigurationFile)
   if jobDistributor.configure():
      exitCode = jobDistributor.processCommandArguments()
      if not exitCode:
         exitCode = jobDistributor.buildJobs()
         if not exitCode:
            jobDistributor.registerJobs()
            jobDistributor.runJobs()
      exitCode = jobDistributor.reportExitCondition()
   else:
      exitCode = 1

   sys.exit(exitCode)


if __name__ == '__main__':
   distributorLogLevel          = os.getenv("DISTRIBUTORLOGLEVEL",DISTRIBUTORLOGLEVEL)
   distributorSysLogLocal       = os.getenv("DISTRIBUTORSYSLOGLOCAL",DISTRIBUTORSYSLOGLOCAL)
   configurationDirectory       = os.getenv("CONFIGURATIONDIRECTORY",CONFIGURATIONDIRECTORY)
   distributorConfigurationFile = os.getenv("DISTRIBUTORCONFIGURATIONFILE",DISTRIBUTORCONFIGURATIONFILE)
   daemonsDirectory             = os.getenv("DAEMONSDIRECTORY",DAEMONSDIRECTORY)
   daemonsConfigurationFile     = os.getenv("DAEMONSCONFIGURATIONFILE",DAEMONSCONFIGURATIONFILE)
   infosDirectory               = os.getenv("INFOSDIRECTORY",INFOSDIRECTORY)
   infosConfigurationFile       = os.getenv("INFOSCONFIGURATIONFILE",INFOSCONFIGURATIONFILE)

   applicationUtilities = ApplicationUtilities(DISTRIBUTORLOGLEVEL,
                                               DISTRIBUTORSYSLOGLOCAL)

   applicationUtilities.openSysLogLogger(distributorSysLogLocal,
                                         distributorLogLevel)

   distributor(configurationDirectory,distributorConfigurationFile,
               daemonsDirectory,daemonsConfigurationFile,
               infosDirectory,infosConfigurationFile)


