#!/usr/bin/env python3
#
# @package      hubzero-submit-distributor
# @file         distributor.py
# @copyright    Copyright (c) 2013-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2013-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
"""Send email notification to HUB user
"""
import sys
import os
import smtplib
import traceback
from email.mime.text import MIMEText
from optparse import OptionParser,SUPPRESS_HELP

HUBMAILROOT = os.path.dirname(os.path.abspath(__file__))
MAXSUBJECTLENGTH = 50
MAXTEXTLENGTH    = 140

class CommandParser:

   def __init__(self):
      self.commandOptions   = None
      self.commandArguments = []

      self.parser = OptionParser(prog="hubmail",add_help_option=False)
      self.parser.disable_interspersed_args()
      self.parser.add_option("-s","--subject",action="store",type="string",dest="subject", \
                                              help="Email subject")
      self.parser.add_option("-t","--text",action="store",type="string",dest="text", \
                                           help="Email message")
      self.parser.set_defaults(help=False)
      self.parser.add_option("-h","--help",action="store_true",dest="help", \
                                           help="Report command usage.")

      self.commandOptions,arguments = self.parser.parse_args(sys.argv[1:])


   def showUsage(self):
      self.parser.print_help()


   def getOption(self,
                 option):
      value = getattr(self.commandOptions,option)

      return(value)


if __name__ == '__main__':

   emailDestination = os.getenv("USER_EMAIL",None)
   emailSource      = os.getenv("HUB_EMAIL_FROM",None)

   commandParser = CommandParser()
   if commandParser.getOption('help'):
      commandParser.showUsage()
   else:
      maxSubjectLength = MAXSUBJECTLENGTH
      maxTextLength    = MAXTEXTLENGTH
      try:
         with open(os.path.join(HUBMAILROOT,'hubmail.cfg')) as fpConfig:
            configOptions = fpConfig.readlines()
      except:
         pass
      else:
         for configOption in configOptions:
            key,value = configOption.split('=')
            key = key.strip()
            if   key == 'maxSubjectLength':
               maxSubjectLength = int(value.strip())
            elif key == 'maxTextLength':
               maxTextLength    = int(value.strip())

      sendMessage = True
      subject = commandParser.getOption('subject')
      text    = commandParser.getOption('text')
      if subject:
         if   len(subject) > maxSubjectLength:
            sys.stderr.write("Subject exceeds maximum length of %d characters\n" % (maxSubjectLength))
            sendMessage = False
         elif len(subject) < 1:
            sys.stderr.write("Subject is empty\n")
            sendMessage = False
      else:
         sys.stderr.write("Subject not specified\n")
         sendMessage = False

      if text:
         if   len(text) > maxTextLength:
            sys.stderr.write("Text exceeds maximum length of %d characters\n" % (maxTextLength))
            sendMessage = False
         elif len(text) < 1:
            sys.stderr.write("Text is empty\n")
            sendMessage = False
      else:
         sys.stderr.write("Text not specified\n")
         sendMessage = False

      if not emailDestination:
         sys.stderr.write("Message destination not specified\n")
         sendMessage = False
      if not emailSource:
         sys.stderr.write("Message source not specified\n")
         sendMessage = False

      if sendMessage:
         try:
            msg = MIMEText(text)
            msg['Subject'] = subject
            msg['From'] = emailSource
            msg['To'] = emailDestination

            try:
               smtp = smtplib.SMTP('localhost')
            except:
               print("connection to mailserver failed")
            else:
               try:
                  result = smtp.sendmail(emailSource,[emailDestination],msg.as_string())
               except Exception:
                  print(traceback.format_exc())

               closed = smtp.quit()
         except Exception:
            print(traceback.format_exc())

