# @package      hubzero-submit-distributor
# @file         JobHarvester.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
__version__ = '3.11.1'

import sys
import os
import re
import signal
import select
import datetime
import time
import pwd
import subprocess
import shlex
import random
import traceback
import math
import copy
import shutil
import logging
import json
from errno import EPIPE

from hubzero.submit.LogMessage            import getLogJobIdMessage as getLogMessage, logSetJobId
from hubzero.submit.DaemonsInfo           import DaemonsInfo
from hubzero.submit.InfosInfo             import InfosInfo
from hubzero.submit.SitesInfo             import SitesInfo
from hubzero.submit.TunnelsInfo           import TunnelsInfo
from hubzero.submit.CommandParser         import CommandParser
from hubzero.submit.HarvestLocal          import HarvestLocal
from hubzero.submit.RemoteIdentityManager import RemoteIdentityManager
from hubzero.submit.RemoteTunnelMonitor   import RemoteTunnelMonitor
from hubzero.submit.JobStatistic          import JobStatistic

POSTPROCESSJOBCOMMAND  = 'postprocessjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

LOCALJOBID  = ".__local_jobid"
REMOTEJOBID = ".__remote_jobid"

TIMESTAMPINPUTBUILT  = ".__timestamp_inputbuilt"
TIMESTAMPINPUTSTAGED = ".__timestamp_inputstaged"
TIMESTAMPTRANSFERRED = ".__timestamp_transferred"
TIMESTAMPFINISH      = ".__timestamp_finish"
TIMESTAMPSTART       = ".__timestamp_start"
TIMERESULTS          = ".__time_results"
HARVESTINFORMATION   = ".__harvest_information"

BATCHQUEUETYPES = ['pbs',
                   'lsf',
                   'll',
                   'sge',
                   'slurm',
                   'condor',
                   'factory',
                   'boinc']

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class JobHarvester:
   def __init__(self,
                configurationDirectory,
                distributorConfigurationFile,
                daemonsDirectory,
                daemonsConfigurationFile,
                infosDirectory,
                infosConfigurationFile):
      self.logger = logging.getLogger(__name__)

      self.configData = {}
      self.configurationDirectory       = configurationDirectory
      self.distributorConfigurationFile = distributorConfigurationFile
      self.daemonsDirectory             = daemonsDirectory
      self.daemonsConfigurationFile     = daemonsConfigurationFile
      self.infosDirectory               = infosDirectory
      self.infosConfigurationFile       = infosConfigurationFile

      self.version              = __version__
      self.session              = 0
      self.jobId                = 0
      self.harvestPath          = ''
      self.jobs                 = {}
      self.jobStatistics        = {}
      self.reportMetrics        = False
      self.childPid             = 0
      self.bufferSize           = 4096
      self.dataDirectory        = ''
      self.binDirectory         = ''

      configFilePath = os.path.join(self.daemonsDirectory,self.daemonsConfigurationFile)
      self.daemonsInfo       = DaemonsInfo(configFilePath)
      self.identityListenURI = self.daemonsInfo.getDaemonListenURI('identitiesManager','tcp')
      self.tunnelListenURI   = self.daemonsInfo.getDaemonListenURI('tunnelMonitor','tcp')

      self.batchCommands = {}
      self.batchCommands['postProcessJob']  = POSTPROCESSJOBCOMMAND
      self.batchCommands['transmitResults'] = TRANSMITRESULTSCOMMAND
      self.batchCommands['cleanupJob']      = CLEANUPJOBCOMMAND
      self.batchCommands['killBatchJob']    = KILLBATCHJOBCOMMAND

      jobIndex = 0
      self.jobStatistics[jobIndex] = JobStatistic(0)

      self.waitForJobsInfo = {}

      self.remoteMonitors = {}
      self.remoteMonitors['identity'] = None
      self.remoteMonitors['tunnel']   = None

      self.infosInfo   = None
      self.sitesInfo   = None
      self.tunnelsInfo = None

      self.successfulInstance        = None
      self.nRedundant                = 1
      self.isMultiCoreRequest        = False
      self.nCpus                     = 1
      self.ppn                       = ""
      self.nGpus                     = 0
      self.gpn                       = 0
      self.memoryMB                  = 0
      self.wallTime                  = 60
      self.x509SubmitProxy           = ""
      self.isParametric              = False
      self.parameterNames            = []
      self.parameterCombinations     = None
      self.parameterCombinationsPath = None
      self.runType                   = None
      self.filesToRemove             = []
      self.emptyFilesToRemove        = []
      self.nInstances                = 0
      self.executeInstance           = None
      self.inputsPath                = None
      self.jobPath                   = None

      self.doubleDashTerminator = False
      self.commandParser        = None

      self.runName             = ""
      self.localJobId          = None
      self.hubUserId           = None
      self.hubUserName         = None
      self.pegasusVersion      = None
      self.clientSudoUser      = None
      self.clientWorkDirectory = None

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def setSystemEnvironment(self):
      os.environ['PATH'] = self.binDirectory + os.pathsep + os.environ['PATH']


   def getSystemEnvironment(self):
      self.hubUserName          = os.getenv("USER")
      self.hubUserId            = os.getuid()
      self.session              = int(os.getenv("SESSION",'0'))
      self.clientSudoUser       = os.getenv("CLIENT_SUDO_USER","")
      self.pegasusVersion       = os.getenv("PEGASUS_VERSION")
      self.clientWorkDirectory  = os.getenv("CLIENT_WORK_DIRECTORY")
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'False').lower() == 'true')


   def configure(self):
      sectionPattern  = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile(r'\s*#.*')
      inDistributorSection = False

      configured = False
      try:
         configFilePath = os.path.join(self.configurationDirectory,self.distributorConfigurationFile)
         fpConfig = open(configFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inDistributorSection = (sectionName == 'distributor')
                     if inDistributorSection:
                        self.configData = {'probeMonitoringInstalled':False,
                                           'maximumSelectedSites':3,
                                           'allowedVenueMechanisms':['local','ssh'],
                                           'dataDirectory':os.path.join(os.sep,'opt','submit'),
                                           'binDirectory':os.path.join(os.sep,'opt','submit','bin'),
                                           'condorRoot':'',
                                           'condorConfig':'',
                                           'pbsRoot':''
                                          }
                  elif inDistributorSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = list(self.configData[key].keys())[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 try:
                                    self.configData[key] = json.loads(value)
                                 except:
                                    self.configData[key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.configData[key] = {}
               
                              if not self.configData[key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
               else:
                  eof = True
                  configured = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (configFilePath)))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (configFilePath)))

      if configured:
         if os.path.isdir(self.configData['dataDirectory']):
            self.dataDirectory = self.configData['dataDirectory']
         else:
            message = "Specified dataDirectory does not exist: %s" % (self.configData['dataDirectory'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            configured = False

         if os.path.isdir(self.configData['binDirectory']):
            self.binDirectory = self.configData['binDirectory']
         else:
            message = "Specified binDirectory does not exist: %s" % (self.configData['binDirectory'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            configured = False

      if configured:
         self.setSystemEnvironment()
         self.getSystemEnvironment()

      return(configured)


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.logger.log(logging.INFO,getLogMessage("Args are:" + str(sys.argv)))
      self.commandParser.parseArguments(sys.argv[1:])

      if self.commandParser.getOption('harvestPath'):
         self.harvestPath = self.commandParser.getOption('harvestPath')
         self.runName     = os.path.basename(self.harvestPath)
      else:
         exitCode = 1
      self.reportMetrics = self.commandParser.getOption('metrics')

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setJobId(self):
      exitCode = 0

      if self.commandParser.getOption('harvestId'):
         self.jobId = self.commandParser.getOption('harvestId')
      if self.jobId > 0:
         self.localJobId = "%08d" % (self.jobId)
         logSetJobId(self.jobId)
      else:
         exitCode = 1

      self.jobStatistics[0]['exitCode'] = exitCode

      return(exitCode)


   def setInfo(self):
      configFilePath = os.path.join(self.infosDirectory,self.infosConfigurationFile)
      self.infosInfo = InfosInfo(configFilePath)

      self.sitesInfo = SitesInfo(self.infosInfo.getInfoPath('sites'),
                                 restrictionUser=os.getenv("USER"),
                                 pegasusTemplateDirectory=self.infosInfo.getInfoPath('pegasusTemplates'),
                                 allowedVenueMechanisms=self.configData['allowedVenueMechanisms'],
                                 pegasusVersion=self.pegasusVersion)

      self.tunnelsInfo = TunnelsInfo(self.infosInfo.getInfoPath('tunnels'))


   def processCommandArguments(self):
      exitCode = self.parseCommandArguments()
      if not exitCode:
         exitCode = self.setJobId()
         if not exitCode:
            self.setInfo()

      return(exitCode)


   def reportExitCondition(self):
      runExitCode = 2
      redundantJobKillStatus = 1 << 7 | signal.SIGUSR1
#     self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(len(self.jobs)): %d" % (len(self.jobs))))
      if len(self.jobs) == 0:
         jobIndices = list(self.jobStatistics.keys())
         maximumJobIndex = max(jobIndices)
#        self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(jobIndices): %s" % (jobIndices)))
#        self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(maximumJobIndex): %d" % (maximumJobIndex)))

         try:
            mechanism      = self.jobStatistics[maximumJobIndex]['jobSubmissionMechanism']
            remoteId       = self.jobStatistics[maximumJobIndex]['remoteJobIdNumber']
            remoteLocation = self.jobStatistics[maximumJobIndex]['venue']
            exitCode       = self.jobStatistics[maximumJobIndex]['exitCode']
            cpuTime        = self.jobStatistics[maximumJobIndex]['userTime']+self.jobStatistics[maximumJobIndex]['sysTime']
            elapsedRunTime = self.jobStatistics[maximumJobIndex]['elapsedRunTime']
            waitTime       = self.jobStatistics[maximumJobIndex]['waitingTime']
            nCpuUsed       = self.jobStatistics[maximumJobIndex]['nCores']
            event          = self.jobStatistics[maximumJobIndex]['event']
            generated      = self.jobStatistics[maximumJobIndex]['generated']

            message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                      (maximumJobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
            if event:
               message += " event=%s" % (event)
            if generated:
               message += " generated=%s" % (generated)
            self.logger.log(logging.INFO,getLogMessage(message))
            message += "\n"
            os.write(3,message.encode('utf-8'))
            runExitCode = exitCode
         except:
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
            pass
      else:
#        if self.successfulInstance == None:
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(runType,successfulInstance): %s None" % \
#                                                                                                   (self.runType)))
#        else:
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(runType,successfulInstance): %s %d" % \
#                                                                           (self.runType,self.successfulInstance)))
         instanceIndex = 0
         instances = list(self.jobs.keys())
         instances.sort()
         for instance in instances:
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(instance): %d" % (instance)))
            if (self.successfulInstance == None) or \
               (self.successfulInstance != None and (instance != self.successfulInstance)):
               jobIndices = list(self.jobs[instance].jobStatistics.keys())
               jobIndices.sort()
               maximumJobIndex = max(jobIndices)
#              self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(jobIndices): %s" % (jobIndices)))
#              self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(maximumJobIndex): %d" % (maximumJobIndex)))

               try:
                  for jobIndex in jobIndices:
                     if jobIndex != 0 or maximumJobIndex == 0:
                        mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                        remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                        remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                        exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                        if jobIndex != maximumJobIndex:
                           if exitCode == 0:
                              exitCode = 65533
                        cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                         self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                        elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                        waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                        nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                        event          = self.jobs[instance].jobStatistics[jobIndex]['event']
                        generated      = self.jobs[instance].jobStatistics[jobIndex]['generated']

                        instanceIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                            (instance,mechanism,remoteId,remoteLocation, \
                                                             exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                        if event:
                           message += " event=%s" % (event)
                        if generated:
                           message += " generated=%s" % (generated)
                        self.logger.log(logging.INFO,getLogMessage(message))
                        message += "\n"
                        os.write(3,message.encode('utf-8'))
                        if   self.runType == 'redundant':
                           if exitCode != redundantJobKillStatus:
                              if runExitCode == 2:
                                 runExitCode = exitCode
                        elif self.runType == 'pegasus':
                           if instance == 0:
                              if exitCode != 0:
                                 runExitCode = exitCode
                           else:
                              if   runExitCode == 2:
                                 runExitCode = exitCode
                              elif runExitCode == 0:
                                 runExitCode = exitCode
                        elif self.runType == 'boinc':
                           if instance == 0:
                              if exitCode != 0:
                                 runExitCode = exitCode
                           else:
                              if   runExitCode == 2:
                                 runExitCode = exitCode
                              elif runExitCode == 0:
                                 runExitCode = exitCode
                        elif self.runType == 'sweep':
                           if   runExitCode == 2:
                              runExitCode = exitCode
                           elif runExitCode == 0:
                              runExitCode = exitCode
               except:
                  self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
                  pass

         if self.successfulInstance != None:
            instance = self.successfulInstance
            jobIndices = list(self.jobs[instance].jobStatistics.keys())
            jobIndices.sort()
            maximumJobIndex = max(jobIndices)
            waitForJobInfo = self.jobs[instance].getWaitForJobInfo()
            executionMode = waitForJobInfo['executionMode']
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(jobIndices): %s" % (jobIndices)))
#           self.logger.log(logging.DEBUG,getLogMessage("reportExitCondition(maximumJobIndex): %d" % (maximumJobIndex)))

            try:
               for jobIndex in jobIndices:
                  if jobIndex != 0 or maximumJobIndex == 0:
                     mechanism      = self.jobs[instance].jobStatistics[jobIndex]['jobSubmissionMechanism']
                     remoteId       = self.jobs[instance].jobStatistics[jobIndex]['remoteJobIdNumber']
                     remoteLocation = self.jobs[instance].jobStatistics[jobIndex]['venue']
                     exitCode       = self.jobs[instance].jobStatistics[jobIndex]['exitCode']
                     if self.jobs[instance].remoteBatchSystem == 'PEGASUS':
                        if mechanism == "" or mechanism == 'Unknown':
                           mechanism = 'PEGASUS'
                     else:
                        if jobIndex != maximumJobIndex:
                           if exitCode == 0:
                              exitCode = 65533
                     cpuTime        = self.jobs[instance].jobStatistics[jobIndex]['userTime'] + \
                                      self.jobs[instance].jobStatistics[jobIndex]['sysTime']
                     elapsedRunTime = self.jobs[instance].jobStatistics[jobIndex]['elapsedRunTime']
                     waitTime       = self.jobs[instance].jobStatistics[jobIndex]['waitingTime']
                     nCpuUsed       = self.jobs[instance].jobStatistics[jobIndex]['nCores']
                     event          = self.jobs[instance].jobStatistics[jobIndex]['event']
                     generated      = self.jobs[instance].jobStatistics[jobIndex]['generated']

                     if instance == 0 and len(jobIndices) == 1 and \
                        (self.runType == 'pegasus' or self.runType == 'boinc'):
                        message = "venue=0:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                          (         mechanism,remoteId,remoteLocation, \
                                                           exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                     else:
                        instanceIndex += 1
                        message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d" % \
                                                          (instance,mechanism,remoteId,remoteLocation, \
                                                           exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
                     if event:
                        message += " event=%s" % (event)
                     if generated:
                        message += " generated=%s" % (generated)
                     message += " executionMode=%s" % (executionMode)
                     self.logger.log(logging.INFO,getLogMessage(message))
                     message += "\n"
                     os.write(3,message.encode('utf-8'))
                     if   self.runType == 'redundant':
                        runExitCode = exitCode
                     elif self.runType == 'pegasus':
                        if instance == 0:
                           if exitCode != 0:
                              runExitCode = exitCode
                     elif self.runType == 'boinc':
                        if instance == 0:
                           if exitCode != 0:
                              runExitCode = exitCode
                     elif self.runType == 'sweep':
                        pass
            except:
               self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
               pass

      message = "runStatus=%d" % (runExitCode)
      self.logger.log(logging.INFO,getLogMessage(message))
      message += "\n"
      os.write(3,message.encode('utf-8'))

      return(runExitCode)


# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      pass


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def executeCommand(self,
                      command,
                      useShell=False,
                      streamOutput=False):
      if isinstance(command,list):
         child = subprocess.Popen(command,shell=useShell,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      else:
         commandArgs = shlex.split(command)
         child = subprocess.Popen(commandArgs,shell=useShell,bufsize=self.bufferSize,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize).decode('utf-8')
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize).decode('utf-8')
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def loadHarvestInformation(self,
                              harvestInformationPath):
      harvestInformation = {'runType':'',
                            'venueMechanism':'',
                            'localJobId':'00000000',
                            'instanceId':0,
                            'nInstances':0,
                            'runName':'',
                            'isParametric':True,
                            'isMultiCoreRequest':False,
                            'nCpus':1,
                            'event':'',
                            'destination':'',
                            'venue':'',
                            'remoteJobIdNumber':'',
                            'stageInTarFile':'',
                            'batchLogPath':'',
                            'batchScriptPath':'',
                            'appScriptPath':'',
                            'nodeFilePath':'',
                            'toolInputTemplateFilePath':'',
                            'toolOutputTemplateFilePath':'',
                            'filesToRemove':[],
                            'emptyFilesToRemove':[],
                            'x509SubmitProxy':''
                           }

      informationLoaded = False
      keyValuePattern = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile(r'\s*#.*')
      try:
         fpHarvestInformation = open(harvestInformationPath,'r')
         try:
            eof = False
            while not eof:
               record = fpHarvestInformation.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in harvestInformation:
                        if   isinstance(harvestInformation[key],list):
                           harvestInformation[key] = [e.strip() for e in value.split(',')]
                        elif isinstance(harvestInformation[key],bool):
                           harvestInformation[key] = bool(value.lower() == 'true')
                        elif isinstance(harvestInformation[key],float):
                           harvestInformation[key] = float(value)
                        elif isinstance(harvestInformation[key],int):
                           harvestInformation[key] = int(value)
                        elif isinstance(harvestInformation[key],dict):
                           try:
                              sampleKey   = harvestInformation[key].keys()[0]
                              sampleValue = harvestInformation[key][sampleKey]
                           except:
                              sampleKey   = "key"
                              sampleValue = "value"
                           harvestInformation[key] = {} 
                           for e in value.split(','):
                              dictKey,dictValue = e.split(':')
                              if isinstance(sampleKey,int):
                                 dictKey = int(dictKey)
                              if   isinstance(sampleValue,int):
                                 dictValue = int(dictValue)
                              elif isinstance(sampleValue,float):
                                 dictValue = float(dictValue)
                              elif isinstance(sampleValue,bool):
                                 dictValue = bool(dictValue.lower() == 'true')
                              harvestInformation[key][dictKey] = dictValue
                        else:
                           harvestInformation[key] = value
                     else:
                        self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
               else:
                  eof = True
                  informationLoaded = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (harvestInformationPath)))
         finally:
            fpHarvestInformation.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (harvestInformationPath)))

      try:
         os.remove(harvestInformationPath)
      except:
         pass

      if not informationLoaded:
         harvestInformation = None

      return(harvestInformation)


   def buildBoincJobInstances(self,
                              currentWorkingDirectory,
                              harvestInformation):
      exitCode = 0
      self.inputsPath      = os.path.join(self.harvestPath,'InputFiles')
      self.nInstances      = harvestInformation['nInstances']
      self.executeInstance = harvestInformation['instanceId']

      timeHistoryLogs = {}
      timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT, \
                                                              self.localJobId,str(self.executeInstance))
      timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED, \
                                                              self.localJobId,str(self.executeInstance))
      timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED, \
                                                              self.localJobId,str(self.executeInstance))
      timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART, \
                                                              self.localJobId,str(self.executeInstance))
      timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH, \
                                                              self.localJobId,str(self.executeInstance))
      timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS, \
                                                              self.localJobId,str(self.executeInstance))

      wfInstanceId = "WF;%d" % (self.nInstances)
      siteInfo = self.sitesInfo.getSiteInfo(harvestInformation['destination'])
      if harvestInformation['venueMechanism'] == 'local':
         self.jobs[self.executeInstance] = HarvestLocal(self.remoteMonitors,
                                                        self.hubUserName,
                                                        self.hubUserId,
                                                        currentWorkingDirectory,
                                                        self.batchCommands,
                                                        self.isParametric,
                                                        self.runName,
                                                        self.localJobId,
                                                        wfInstanceId,
                                                        harvestInformation,
                                                        siteInfo,
                                                        timeHistoryLogs)

      nInstanceIdDigits = max(2,int(math.log10(max(1,self.nInstances))+1))
      for instance in iterRange(1,self.nInstances+1):
         instanceDirectory = str(instance).zfill(nInstanceIdDigits)
         harvestInformationPath = os.path.join(self.harvestPath,instanceDirectory,"%s.%s_%s" % (HARVESTINFORMATION, \
                                                                              self.localJobId,instanceDirectory))
         harvestInstanceInformation = self.loadHarvestInformation(harvestInformationPath)
         if harvestInstanceInformation:
            instanceId = harvestInstanceInformation['instanceId']

            timeHistoryLogs = {}
            timeHistoryLogs['timestampInputBuilt']  = "%s.%s_%s" % (TIMESTAMPINPUTBUILT, \
                                                                    self.localJobId,str(instanceDirectory))
            timeHistoryLogs['timestampInputStaged'] = "%s.%s_%s" % (TIMESTAMPINPUTSTAGED, \
                                                                    self.localJobId,str(instanceDirectory))
            timeHistoryLogs['timestampTransferred'] = "%s.%s_%s" % (TIMESTAMPTRANSFERRED, \
                                                                    self.localJobId,str(instanceDirectory))
            timeHistoryLogs['timestampStart']       = "%s.%s_%s" % (TIMESTAMPSTART, \
                                                                    self.localJobId,str(instanceDirectory))
            timeHistoryLogs['timestampFinish']      = "%s.%s_%s" % (TIMESTAMPFINISH, \
                                                                    self.localJobId,str(instanceDirectory))
            timeHistoryLogs['timeResults']          = "%s.%s_%s" % (TIMERESULTS, \
                                                                    self.localJobId,str(instanceDirectory))

            if harvestInstanceInformation['venueMechanism'] == 'local':
               self.jobs[instance] = HarvestLocal(self.remoteMonitors,
                                                  self.hubUserName,
                                                  self.hubUserId,
                                                  currentWorkingDirectory,
                                                  self.batchCommands,
                                                  self.isParametric,
                                                  self.runName,
                                                  self.localJobId,
                                                  instanceDirectory,
                                                  harvestInstanceInformation,
                                                  siteInfo,
                                                  timeHistoryLogs)
      return(exitCode)


   def buildJobInstances(self):
      exitCode = 0
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("buildJobInstances:os.getcwd(): No such file or directory"))
         exitCode = 1
      else:
         self.remoteMonitors['identity'] = RemoteIdentityManager(self.identityListenURI)
         self.remoteMonitors['tunnel']   = RemoteTunnelMonitor(self.tunnelListenURI)

         harvestInformationPath = os.path.join(self.harvestPath,"%s.%s_%s" % (HARVESTINFORMATION, \
                                                                              self.localJobId,'0'))
         harvestInformation = self.loadHarvestInformation(harvestInformationPath)
         if harvestInformation:
            self.runType = harvestInformation['runType']
            if 'filesToRemove' in harvestInformation:
               self.filesToRemove      = copy.copy(harvestInformation['filesToRemove'])
            if 'emptyFilesToRemove' in harvestInformation:
               self.emptyFilesToRemove = copy.copy(harvestInformation['emptyFilesToRemove'])

            if   self.runType == 'boinc':
               exitCode = self.buildBoincJobInstances(currentWorkingDirectory,
                                                      harvestInformation)


#           if   self.runType == 'redundant':
#              exitCode = self.buildRedundantJobInstances(currentWorkingDirectory)
#           elif self.runType == 'pegasus':
#              exitCode = self.buildPegasusJobInstances(currentWorkingDirectory)
#           elif self.runType == 'boinc':
#              exitCode = self.buildBoincJobInstances(currentWorkingDirectory)
#           elif self.runType == 'sweep':
#              exitCode = self.buildSweepJobInstances(currentWorkingDirectory)
            else:
               self.logger.log(logging.ERROR,getLogMessage("buildJobInstances: runType has not been properly set."))
               exitCode = 1
         else:
            exitCode = 1

      return(exitCode)


   def harvestRedundantJobInstances(self):
      nBatchJobsHeld = self.nInstances
      jobsRunning = False
      if self.successfulInstance == None:
         while (jobsRunning or nBatchJobsHeld > 0) and (self.successfulInstance == None):
            completeRemoteJobIndexes = self.remoteMonitors['job'].waitForRedundantJobs(self.waitForJobsInfo,
                                                                                       self.progressReport,
                                                                                       self.abortGlobal)

            nBatchJobsHeld = 0
            jobsRunning = False
            for instance in self.waitForJobsInfo:
               if self.waitForJobsInfo[instance]['state'] == 'released':
                  if 'recentJobSite' in self.waitForJobsInfo[instance]:
                     executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                     self.jobs[instance].updateVenue(executionVenue)
                     if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                        jobsRunning = True
               else:
                  nBatchJobsHeld += 1

            for instance in completeRemoteJobIndexes:
               self.jobs[instance].postProcess()
               self.jobs[instance].retrieveFiles()
               self.jobs[instance].cleanupFiles()
               self.jobs[instance].recordJobStatistics()
               if self.jobs[instance].wasJobSuccessful():
                  self.successfulInstance = instance

      if self.successfulInstance != None:
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if   self.waitForJobsInfo[instance]['recentJobStatus'] == 'KD':
                  self.jobs[instance].cleanupFiles()
                  self.waitForJobsInfo[instance]['recentJobStatus'] = 'D'
               elif instance != self.successfulInstance:
                  self.jobs[instance].cleanupFiles()

      if self.successfulInstance != None:
         self.jobs[self.successfulInstance].recoverFiles(False)
         fileToMove = "%s_%02d.stdout" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stdout" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.stderr" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.stderr" % (self.runName)
            os.rename(fileToMove,fileNewName)
         fileToMove = "%s_%02d.SUCCESS" % (self.runName,self.successfulInstance)
         if os.path.exists(fileToMove):
            fileNewName = "%s.SUCCESS" % (self.runName)
            os.rename(fileToMove,fileNewName)
      else:
         for instance in self.waitForJobsInfo:
            self.jobs[instance].recoverStdFiles()
         if len(self.jobs) > 0:
            recoverInstance = random.choice(self.jobs.keys())
            self.jobs[recoverInstance].recoverFiles(True)

      for instance in self.waitForJobsInfo:
         self.jobs[instance].removeInstanceDirectory()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)


   def harvestPegasusJobInstances(self):
      nBatchJobsHeld = 1
      jobsRunning = False
      while (jobsRunning or nBatchJobsHeld > 0) and (self.successfulInstance == None):
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForPegasusWorkflowJobs(self.waitForJobsInfo,
                                                                                          self.nInstances,
                                                                                          self.parameterCombinationsPath,
                                                                                          self.progressReport,self.abortGlobal)
         nBatchJobsHeld = 0
         jobsRunning = False
         for instance in [self.executeInstance]:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()
            if self.jobs[instance].wasJobSuccessful():
               self.successfulInstance = instance

      if self.successfulInstance != None:
         for instance in iterRange(1,self.nInstances+1):
            self.jobs[instance].jobSubmitted = True

      for instance in iterRange(1,self.nInstances+1):
         if 'recentJobSite' in self.waitForJobsInfo[instance]:
            executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
            self.jobs[instance].updateVenue(executionVenue)
         self.jobs[instance].postProcess()
         self.jobs[instance].retrieveFiles()
         self.jobs[instance].cleanupFiles()
         self.jobs[instance].recordJobStatistics()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if self.inputsPath:
         if os.path.isdir(self.inputsPath):
            shutil.rmtree(self.inputsPath,True)


   def harvestBoincJobInstances(self):
      for instance in [self.executeInstance]:
         self.jobs[instance].postProcess()
         self.jobs[instance].retrieveFiles()
         self.jobs[instance].cleanupFiles()
         self.jobs[instance].recordJobStatistics()
         if self.jobs[instance].wasJobSuccessful():
            self.successfulInstance = instance

      if self.successfulInstance != None:
         for instance in iterRange(1,self.nInstances+1):
            try:
               self.jobs[instance].jobSubmitted = True
            except:
               self.logger.log(logging.ERROR,getLogMessage("harvestBoincJobInstances: missing instance %d" % (instance)))

      for instance in iterRange(1,self.nInstances+1):
         try:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()
         except:
            pass

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if self.inputsPath:
         if os.path.isdir(self.inputsPath):
            shutil.rmtree(self.inputsPath,True)


   def harvestSweepJobInstances(self):
      nBatchJobsHeld = self.nInstances
      jobsRunning = False
      while (jobsRunning or nBatchJobsHeld > 0):
         completeRemoteJobIndexes = self.remoteMonitors['job'].waitForSweepJobs(self.waitForJobsInfo,
                                                                                self.parameterCombinationsPath,
                                                                                self.progressReport,self.abortGlobal)
         nBatchJobsHeld = 0
         jobsRunning = False
         for instance in self.waitForJobsInfo:
            if self.waitForJobsInfo[instance]['state'] == 'released':
               if 'recentJobSite' in self.waitForJobsInfo[instance]:
                  executionVenue = self.waitForJobsInfo[instance]['recentJobSite']
                  self.jobs[instance].updateVenue(executionVenue)
                  if self.waitForJobsInfo[instance]['recentJobStatus'] != 'D':
                     jobsRunning = True
            else:
               nBatchJobsHeld += 1

         for instance in completeRemoteJobIndexes:
            self.jobs[instance].postProcess()
            self.jobs[instance].retrieveFiles()
            self.jobs[instance].cleanupFiles()
            self.jobs[instance].recordJobStatistics()

      for fileToRemove in self.filesToRemove:
         if os.path.exists(fileToRemove):
            os.remove(fileToRemove)

      for fileToRemove in self.emptyFilesToRemove:
         if os.path.exists(fileToRemove):
            if(os.path.getsize(fileToRemove) == 0):
               os.remove(fileToRemove)

      if self.inputsPath:
         if os.path.isdir(self.inputsPath):
            shutil.rmtree(self.inputsPath,True)


   def harvestJobInstances(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("harvestJobInstances:os.getcwd(): No such file or directory"))
      else:
         if   self.runType == 'boinc':
            self.harvestBoincJobInstances()
#        if   self.runType == 'redundant':
#           self.harvestRedundantJobInstances()
#        elif self.runType == 'pegasus':
#           self.harvestPegasusJobInstances()
#        elif self.runType == 'boinc':
#           self.harvestBoincJobInstances()
#        elif self.runType == 'sweep':
#           self.harvestSweepJobInstances()


