# @package      hubzero-submit-distributor
# @file         HarvestCore.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import select
import subprocess
import shlex
import time
import datetime
import re
import shutil
import copy
import logging
from errno import EPIPE

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic import JobStatistic

class HarvestCore:
   def __init__(self,
                harvestInfo,
                siteInfo,
                remoteMonitors,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.timeHistoryLogs       = copy.copy(timeHistoryLogs)
      try:
         self.event              = harvestInfo['event']
      except:
         self.event              = ""
      self.siteInfo              = copy.copy(siteInfo)
      self.remoteTunnelMonitor   = remoteMonitors['tunnel']
      self.remoteIdentityManager = remoteMonitors['identity']

      self.childPid   = 0
      self.bufferSize = 4096

      self.venueMechanism    = ''
      self.remoteBatchSystem = ''
      try:
         self.destination = harvestInfo['destination']
      except:
         self.destination = ""
      try:
         self.venue = harvestInfo['venue']
      except:
         self.venue = ""

      try:
         self.isMultiCoreRequest = harvestInfo['isMultiCoreRequest']
      except:
         self.isMultiCoreRequest = False
      try:
         self.nCpus = harvestInfo['nCpus']
      except:
         self.nCpus = 1

      self.jobSubmitted  = True
      self.jobStatistics = {}
      self.jobIndex      = 0
      if self.isMultiCoreRequest:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      else:
         self.jobStatistics[self.jobIndex] = JobStatistic(1)

      self.jobPostProcessed  = False
      self.filesRetrieved    = False
      self.filesCleanedup    = False
      self.runName           = ""
      self.localJobId        = ""
      self.instanceId        = "-1"
      self.instanceDirectory = ""
      if 'stageInTarFile' in harvestInfo:
         if harvestInfo['stageInTarFile'].endswith('.gz'):
            self.stageInTarFile = harvestInfo['stageInTarFile']
         else:
            self.stageInTarFile = harvestInfo['stageInTarFile'] + '.gz'
      else:
         self.stageInTarFile    = ""
      try:
         self.x509SubmitProxy = harvestInfo['x509SubmitProxy']
      except:
         self.x509SubmitProxy = ""

      try:
         self.batchLogPath = harvestInfo['batchLogPath']
      except:
         self.batchLogPath = ""
      try:
         self.appScriptPath = harvestInfo['appScriptPath']
      except:
         self.appScriptPath = ""
      try:
         self.batchScriptPath = harvestInfo['batchScriptPath']
      except:
         self.batchScriptPath = ""
      try:
         self.nodeFilePath = harvestInfo['nodeFilePath']
      except:
         self.nodeFilePath = ""
      try:
         self.toolInputTemplateFilePath = harvestInfo['toolInputTemplateFilePath']
      except:
         self.toolInputTemplateFilePath = ""
      try:
         self.toolOutputTemplateFilePath = harvestInfo['toolOutputTemplateFilePath']
      except:
         self.toolOutputTemplateFilePath = ""
      try:
         self.remoteJobIdNumber = harvestInfo['remoteJobIdNumber']
      except:
         self.remoteJobIdNumber = ""

   def _writeToStdout(self,
                      message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def _writeToStderr(self,
                      message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError as e:
         if not e.args[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def executeCommand(self,
                      command,
                      streamOutput=False):
      child = subprocess.Popen(command,shell=True,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize).decode('utf-8')
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self._writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize).decode('utf-8')
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self._writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def executeSSHCommand(self,
                         sshCommand,
                         tunnelDesignator,
                         streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.incrementTunnelUse(tunnelDesignator)
      exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

      while exitStatus and (stdError.count("You don't exist, go away!") > 0):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(sshCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      if tunnelDesignator != "":
         self.remoteTunnelMonitor.decrementTunnelUse(tunnelDesignator)

      return(exitStatus,stdOutput,stdError)


   def executeLaunchCommand(self,
                            launchCommand,
                            streamOutput=False):

      minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
      maximumDelay = 256
      updateFrequency = 1
      maximumDelayTime = 900

      delayTime = 0
      sleepTime = minimumDelay
      nDelays = 0
      exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

      while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
         nDelays += 1
         time.sleep(sleepTime)
         delayTime += sleepTime
         if nDelays == updateFrequency:
            nDelays = 0
            sleepTime *= 2
            if sleepTime > maximumDelay:
               sleepTime = maximumDelay

         exitStatus,stdOutput,stdError = self.executeCommand(launchCommand,streamOutput)

         if delayTime >= maximumDelayTime:
            break

      return(exitStatus,stdOutput,stdError)


   def updateVenue(self,
                   executionVenue):
      if self.remoteBatchSystem == 'FACTORY':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue
      if self.remoteBatchSystem == 'PEGASUS':
         self.jobStatistics[self.jobIndex]['venue'] = executionVenue


   def getVenue(self):
      executionVenue = self.jobStatistics[self.jobIndex]['venue']

      return(executionVenue)


   def recordJobStatisticTime(self,
                              statistic,
                              timeFileBasename):
      timePath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timePath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTime: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTime(statistic,timePath)
      else:
         self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))


   def recordJobStatisticTimer(self,
                               timeFileBasename):
      timerPath = os.path.join(self.instanceDirectory,timeFileBasename)
      if os.path.exists(timerPath):
         jobIndex = int(self.instanceId)

         if not jobIndex in self.jobStatistics:
            self.jobStatistics[jobIndex] = JobStatistic(self.nCpus)
            self.logger.log(logging.DEBUG,getLogMessage("recordJobStatisticTimer: added missing jobIndex %d" % (jobIndex)))

         self.jobStatistics[jobIndex].recordTimer(timerPath)
      else:
         if self.jobSubmitted:
            self.logger.log(logging.ERROR,getLogMessage(timeFileBasename + " not present in " + self.instanceDirectory))

# files of this form are generated by non parametric dax execution
      reFiles = re.compile(timeFileBasename + "_[0-9]+$")
      dirFiles = os.listdir(self.instanceDirectory)
      matchingFiles = list(filter(reFiles.search,dirFiles))
      if len(matchingFiles) > 0:
         maximumJobIndex = max(self.jobStatistics.keys())
         jobIndex = maximumJobIndex
         for matchingFile in matchingFiles:
            jobIndex += 1
            self.jobStatistics[jobIndex] = JobStatistic(1)
            timerPath = os.path.join(self.instanceDirectory,matchingFile)
            self.jobStatistics[jobIndex].recordTimer(timerPath)


   def recordJobStatistics(self):
      self.recordJobStatisticTimer(self.timeHistoryLogs['timeResults'])

      if self.jobSubmitted:
         self.recordJobStatisticTime('transferCompleteTime',self.timeHistoryLogs['timestampTransferred'])
         self.recordJobStatisticTime('jobStartedTime',self.timeHistoryLogs['timestampStart'])
         self.recordJobStatisticTime('jobFinshedTime',self.timeHistoryLogs['timestampFinish'])

      for jobIndex in self.jobStatistics:
         self.jobStatistics[jobIndex].setWaitingTime()
         self.jobStatistics[jobIndex].setElapsedRunTime()


   def wasJobSuccessful(self):
      success = False
      for jobIndex in self.jobStatistics:
         if self.jobStatistics[jobIndex]['exitCode'] == 0:
            success = True
            break

      return(success)


   def removeJobStatisticsFiles(self):
      for timeFileBasename in self.timeHistoryLogs['timeResults'], \
                              self.timeHistoryLogs['timestampTransferred'], \
                              self.timeHistoryLogs['timestampStart'], \
                              self.timeHistoryLogs['timestampFinish']:
         reFiles = re.compile(timeFileBasename + "(_[0-9]+)?$")
         dirFiles = os.listdir(self.instanceDirectory)
         matchingFiles = list(filter(reFiles.search,dirFiles))
         if len(matchingFiles) > 0:
            for matchingFile in matchingFiles:
               try:
                  os.remove(os.path.join(self.instanceDirectory,matchingFile))
               except:
                  pass


   def removeInstanceDirectory(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("removeInstanceDirectory:os.getcwd(): No such file or directory"))
      else:
         if self.instanceDirectory != currentWorkingDirectory:
            shutil.rmtree(self.instanceDirectory,True)
            jobDirectory = os.path.dirname(self.instanceDirectory)
            if jobDirectory != currentWorkingDirectory:
               try:
                  os.rmdir(jobDirectory)
               except:
                  pass


   def moveTree(self,
                src,
                dst,
                symlinks=False):
      if os.path.isdir(src):
         if os.path.exists(dst):
            if not os.path.isdir(dst):
               self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (dst)))
               return
         else:
            os.mkdir(dst)
         names = os.listdir(src)
         for name in names:
            srcPath = os.path.join(src,name)
            dstPath = os.path.join(dst,name)
            try:
               if symlinks and os.path.islink(srcPath):
                  linkto = os.readlink(srcPath)
                  os.symlink(linkto,dstPath)
               elif os.path.isdir(srcPath):
                  self.moveTree(srcPath,dstPath,symlinks)
               else:
                  os.rename(srcPath,dstPath)
            except (IOError,os.error) as why:
               self.logger.log(logging.ERROR,getLogMessage("moveTree: Can't move %s to %s: %s" % (srcPath,dstPath,str(why))))
      else:
         self.logger.log(logging.ERROR,getLogMessage("moveTree: %s must be a directory" % (src)))


   def recoverFiles(self,
                    ingoreInstanceFiles):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.exists(self.instanceDirectory):
            if os.path.isdir(self.instanceDirectory):
               if self.instanceDirectory != currentWorkingDirectory:
                  dirFiles = os.listdir(self.instanceDirectory)
                  if ingoreInstanceFiles:
                     ignoreFiles = fnmatch.filter(dirFiles,"*%s_%s*" % (self.runName,self.instanceId))
                     for ignoreFile in ignoreFiles:
                        dirFiles.remove(ignoreFile)

                  for dirFile in dirFiles:
                     srcFile = os.path.join(self.instanceDirectory,dirFile)
                     dstFile = os.path.join(currentWorkingDirectory,dirFile)
                     try:
                        if os.path.isdir(srcFile):
                           self.moveTree(srcFile,dstFile)
                        else:
                           os.rename(srcFile,dstFile)
                     except:
                        self.logger.log(logging.ERROR,getLogMessage("%s recovery failed" % (srcFile)))


   def recoverStdFiles(self):
      try:
         currentWorkingDirectory = os.getcwd()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("recoverStdFiles:os.getcwd(): No such file or directory"))
      else:
         if os.path.isdir(self.instanceDirectory):
            if self.instanceDirectory != currentWorkingDirectory:
               for fileType in 'stderr','stdout','FAILURE':
                  dirFiles = os.listdir(self.instanceDirectory)
                  matchingFiles = fnmatch.filter(dirFiles,"*%s_%s.%s" % (self.runName,self.instanceId,fileType))
                  if len(matchingFiles) > 0:
                     matchingFiles.sort(reverse=True)
                     fileToAppend = "%s.%s" % (self.runName,fileType)
                     try:
                        fpAppend = open(fileToAppend,'ab')
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToAppend)))
                     else:
                        for matchingFile in matchingFiles:
                           fileToRecover = os.path.join(self.instanceDirectory,matchingFile)
                           try:
                              fpRecover = open(fileToRecover,'rb')
                           except (IOError,OSError):
                              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (fileToRecover)))
                           else:
                              try:
                                 shutil.copyfileobj(fpRecover,fpAppend)
                              except:
                                 self.logger.log(logging.ERROR,getLogMessage("%s could not be appended to %s" % \
                                                                                 (fileToRecover,fileToAppend)))
                              finally:
                                 fpRecover.close()
                        fpAppend.close()


   def cleanupPegasusFiles(self):
      for ftype in '.dax','_sites.xml','_sites.yml','.pegasusrc','_tc.txt','_tc.yml':
         pegasusFile = os.path.join(self.instanceDirectory,"%s_%s%s" % (self.localJobId,self.instanceId,ftype))
         if os.path.isfile(pegasusFile):
            try:
               os.remove(pegasusFile)
            except:
               pass
         else:
            pegasusFile = os.path.join(self.instanceDirectory,"%s%s" % (self.localJobId,ftype))
            if os.path.isfile(pegasusFile):
               try:
                  os.remove(pegasusFile)
               except:
                  pass

      pegasusFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.localJobId,self.instanceId))
      if os.path.exists(pegasusFile):
         if(os.path.getsize(pegasusFile) == 0):
            try:
               os.remove(pegasusFile)
            except:
               pass


   def cleanupScriptTemplateLogFiles(self):
      for scriptTemplateLogFile in [self.batchScriptPath,self.appScriptPath,self.nodeFilePath,
                                    self.toolInputTemplateFilePath,self.toolOutputTemplateFilePath,
                                    self.batchLogPath]:
         if os.path.isfile(scriptTemplateLogFile):
            os.remove(scriptTemplateLogFile)


   def cleanupStageInTarFile(self):
      stageInTarFile = os.path.join(self.instanceDirectory,self.stageInTarFile)
      if os.path.isfile(stageInTarFile):
         try:
            os.remove(stageInTarFile)
         except:
            pass

      stageInTarFile += '.gz'
      if os.path.isfile(stageInTarFile):
         try:
            os.remove(stageInTarFile)
         except:
            pass


   def cleanupStageOutTarFile(self):
      stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
      if os.path.isfile(stageOutTarFile):
         try:
            os.remove(stageOutTarFile)
         except:
            pass

      stageOutTarFile += '.gz'
      if os.path.isfile(stageOutTarFile):
         try:
            os.remove(stageOutTarFile)
         except:
            pass


