# @package      hubzero-submit-distributor
# @file         VenueMechanismCache.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import re
import stat
import shutil
import time
import datetime
import logging

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.JobOutput              import JobOutput
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore

CACHEHITMARKER = ".__cacheHit_Marker"

class VenueMechanismCache(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                submitterClass,
                currentWorkingDirectory,
                session,
                distributorPid,
                batchCommands,
                isParametric,
                runName,
                localJobId,
                instanceId,
                destination,
                tailFiles,
                enteredCommand,
                gridsite,
                pegasusSiteInfo,
                stageInTarFile,
                auxiliaryTarFile,
                transferExecutable,
                executable,
                appsAccessInfo,
                event,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                disableJobMonitoring,
                siteInfo,
                tapis2SiteInfo,
                tapis3SiteInfo,
                fileMoverInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                x509SubmitProxy,
                disableProbeCheck,
                timeHistoryLogs,
                useSetup,
                pegasusVersion,
                pegasusHome):
      VenueMechanismCore.__init__(self,session,hubUserName,hubUserId,submitterClass,distributorPid,
                                       timeHistoryLogs,disableJobMonitoring,
                                       siteInfo,tapis2SiteInfo,tapis3SiteInfo,fileMoverInfo,
                                       toolFilesInfo,dockerImageInfo,submissionScriptsInfo,managerInfo,
                                       remoteMonitors,isMultiCoreRequest,nCpus,nNodes,ppn,event)

      self.logger               = logging.getLogger(__name__)
      self.jobOutput            = JobOutput()
      self.venueMechanism       = 'http'
      self.runName              = runName
      self.localJobId           = localJobId
      self.instanceId           = instanceId
      self.nInstances           = 1
      self.remoteBatchSystem    = 'CACHE'
      self.destination          = destination
      self.venue                = siteInfo['venue']
      self.executionMode        = siteInfo['executionMode']

      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile      = stageInTarFile
      else:
         if siteInfo['stageFiles']:
            self.stageInTarFile   = stageInTarFile + '.gz'
         else:
            self.stageInTarFile   = stageInTarFile
      if auxiliaryTarFile.endswith('.gz'):
         self.auxiliaryTarFile    = auxiliaryTarFile
      else:
         if siteInfo['stageFiles']:
            self.auxiliaryTarFile = auxiliaryTarFile + '.gz'
         else:
            self.auxiliaryTarFile = auxiliaryTarFile

      self.isBatchJob        = False
      self.instanceDirectory = os.path.join(currentWorkingDirectory,self.runName,self.instanceId)
      if not os.path.isdir(self.instanceDirectory):
         os.makedirs(self.instanceDirectory)

      self.workingDirectory = currentWorkingDirectory
      self.scratchDirectory = self.instanceDirectory


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      self.logger.log(logging.INFO,getLogMessage("workingDirectory " + self.workingDirectory))
      self.remoteAppScript = None
      self.remoteBatch     = None

      cacheHitMarkerFile = "%s.%s_%s" % (CACHEHITMARKER,self.localJobId,self.instanceId)
      cacheHitMarkerPath = os.path.join(self.instanceDirectory,cacheHitMarkerFile)
      touchCommand = "touch " + cacheHitMarkerPath
      self.logger.log(logging.INFO,getLogMessage("command = " + touchCommand))
      touchExitStatus,touchStdOutput,touchStdError = self.executeCommand(touchCommand)
      if touchExitStatus:
         self.logger.log(logging.ERROR,getLogMessage(touchStdError))
         self.logger.log(logging.ERROR,getLogMessage(touchStdOutput))
         self.__writeToStderr(touchStdError)
         exitCode = 1
      else:
         scriptFiles.append(cacheHitMarkerPath)

      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = int(self.instanceId)
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism
      if self.event:
         self.jobStatistics[self.jobIndex]['event'] = self.event

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      self.jobStatistics[self.jobIndex]['venue'] = self.venue

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      self.removeJobRegistration()

      self.jobSubmitted = False
      self.jobStatistics[self.jobIndex]['exitCode'] = 0

      return(True)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['knownSite']             = self.destination
      waitForJobInfo['siteMonitorDesignator'] = ""
      waitForJobInfo['executionMode']         = self.executionMode
      waitForJobInfo['instanceToken']         = self.instanceToken

      return(waitForJobInfo)


   def postProcess(self):
      self.jobPostProcessed = True


   def retrieveFiles(self):
      stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
      stageOutTarPath = os.path.join(self.instanceDirectory,stageOutTarFile)
      if os.path.exists(stageOutTarPath):
         command = "tar xmf " + stageOutTarPath + \
                            " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         exitStatus,stdOutput,stdError = self.executeCommand(command)
         if exitStatus == 0:
            try:
               os.remove(stageOutTarPath)
            except:
               pass
         else:
            self.jobStatistics[self.jobIndex]['exitCode'] = 12
            if stdOutput != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdOutput)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

            if stdError != "":
               if self.instanceId != "0":
                  stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
               else:
                  stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
               try:
                  fpStd = open(stdFile,'a')
                  try:
                     fpStd.write(stdError)
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                  finally:
                     fpStd.close()
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         if not exitStatus:
            self.filesRetrieved = True


   def cleanupFiles(self):
      self.cleanupStageInTarFile()
      self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 7 | signalNumber

      self.scriptsKilled = True


