# @package      hubzero-submit-distributor
# @file         RemoteBatchAppScript.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import logging
import socket

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteBatchAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                session,
                instanceToken,
                wsJobId,
                runName,
                localJobId,
                instanceId,
                isPegasus,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.session               = session
      self.instanceToken         = instanceToken
      self.wsJobId               = wsJobId
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.isPegasus             = isPegasus
      self.sharedUserSpace       = siteInfo['sharedUserSpace']
      self.stageFiles            = siteInfo['stageFiles']
      self.timePaths             = siteInfo['timePaths']
      self.submissionScriptsInfo = submissionScriptsInfo
      self.transferExecutable    = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable      = os.path.join(".",os.path.basename(executable))
         else:
            self.executable      = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable      = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable      = executable
      self.stdinput              = stdinput
      self.arguments             = arguments
      self.useEnvironment        = useEnvironment
      self.environment           = environment
      self.isMultiCoreRequest    = isMultiCoreRequest
      self.computationMode       = managerInfo['computationMode']
      self.mpiRankVariable       = managerInfo['mpiRankVariable']
      self.preManagerCommands    = managerInfo['preManagerCommands']
      self.managerCommand        = managerInfo['managerCommand']
      self.postManagerCommands   = managerInfo['postManagerCommands']
      self.timestampTransferred  = timeHistoryLogs['timestampTransferred']
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timeResults           = timeHistoryLogs['timeResults']


   def __buildAppSerialScript(self,
                              doInputOutputPacking):
      if self.isPegasus:
         if doInputOutputPacking:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','pegasuspacking')
         else:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','pegasus')
      else:
         if doInputOutputPacking:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','serialpacking')
         else:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','serial')

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"
      useEnvironmentExport = self.useEnvironment

      substitutions = {}
      substitutions["SESSION"]             = self.session
      substitutions["INSTANCETOKEN"]       = self.instanceToken
      substitutions["WSJOBID"]             = self.wsJobId
      substitutions["RUNNAME"]             = self.runName
      substitutions["JOBID"]               = self.localJobId
      substitutions["INSTANCEID"]          = self.instanceId
      substitutions["EXECUTABLE"]          = self.executable
      if self.stdinput == "":
         substitutions["STDIN"]            = '/dev/null'
      else:
         substitutions["STDIN"]            = self.stdinput
      substitutions["ARGUMENTS"]           = self.arguments
      substitutions["USEENVIRONMENT"]      = useEnvironmentExport
      substitutions["ENVIRONMENT"]         = environmentExport
      substitutions["TS_TRANSFERRED"]      = self.timestampTransferred
      substitutions["TS_START"]            = self.timestampStart
      substitutions["TS_FINISH"]           = self.timestampFinish
      substitutions["TIME_RESULTS"]        = self.timeResults
      substitutions["PREMANAGERCOMMANDS"]  = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]      = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"] = "\n".join(self.postManagerCommands)
      substitutions["HUBUSERNAME"]         = self.hubUserName
      substitutions["HUBUSERID"]           = str(self.hubUserId)
      substitutions["SUBMITHOSTFQDN"]      = socket.getfqdn()
      substitutions["TIMEPATHS"]           = ' '.join(self.timePaths)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildAppMPIScript(self,
                           doInputOutputPacking):
      if doInputOutputPacking:
         if self.mpiRankVariable == "":
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','mpinorankpacking')
         else:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','mpipacking')
      else:
         if self.mpiRankVariable == "":
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','mpinorank')
         else:
            rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','APP','mpi')

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"
      useEnvironmentExport = self.useEnvironment

      substitutions = {}
      substitutions["SESSION"]         = self.session
      substitutions["INSTANCETOKEN"]   = self.instanceToken
      substitutions["WSJOBID"]         = self.wsJobId
      substitutions["RUNNAME"]         = self.runName
      substitutions["JOBID"]           = self.localJobId
      substitutions["INSTANCEID"]      = self.instanceId
      substitutions["EXECUTABLE"]      = self.executable
      if self.stdinput == "":
         substitutions["STDIN"]        = '/dev/null'
      else:
         substitutions["STDIN"]        = self.stdinput
      substitutions["ARGUMENTS"]       = self.arguments
      substitutions["USEENVIRONMENT"]  = useEnvironmentExport
      substitutions["ENVIRONMENT"]     = environmentExport
      substitutions["MPIRANKVARIABLE"] = self.mpiRankVariable
      substitutions["TIME_RESULTS"]    = self.timeResults
      substitutions["HUBUSERNAME"]     = self.hubUserName
      substitutions["HUBUSERID"]       = str(self.hubUserId)
      substitutions["TIMEPATHS"]       = ' '.join(self.timePaths)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      doInputOutputPacking = self.sharedUserSpace and self.stageFiles
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript(doInputOutputPacking)
         elif self.computationMode == 'mpiNoFork':
            appScript = ""
         elif self.computationMode == 'parallel':
            appScript = ""
         elif self.computationMode == 'matlabmpi':
            appScript = ""
      else:
         appScript = self.__buildAppSerialScript(doInputOutputPacking)
      appScriptExecutable = True

      return(appScriptName,appScript,appScriptExecutable)


