#
# @package      hubzero-submit-distributor
# @file         Scripts/tapis3/TapisAPI.py
# @copyright    Copyright 2024-2025 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2024-2025 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import sys
import stat
import re
import json
import datetime
from dateutil.tz import tzoffset
from operator import attrgetter
import shutil
import time

from tapipy.tapis  import Tapis
from tapipy.tapis  import TapisResult
from tapipy.util   import tapisresult_to_json_serializer
from tapipy.errors import BaseTapyException

BASEDIR                = os.path.dirname(os.path.abspath(__file__))
CONFIGURATIONDIRECTORY = os.path.join(BASEDIR,'Configuration')

class TapisAPI(Tapis):
   def __init__(self,
                configurationDirectory=CONFIGURATIONDIRECTORY,
                authenticationMode='accessOnly'):
      self.configurationDirectory = configurationDirectory
      self.authenticationMode     = authenticationMode

      self.configData = {}
      self.configured = self.__configure()

      if self.configured:
         if   authenticationMode == 'accessOnly':
            self.__getAccessOnlySession()
         elif authenticationMode == 'clientManager':
            self.__getClientManagerSession()
         elif authenticationMode == 'resetToken':
            self.__resetToken()


   def __configure(self):
      configured = False

      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inTapisSection = False

      configurationFilePath = os.path.join(self.configurationDirectory,'tapis.conf')
      try:
         fpConfig = open(configurationFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inTapisSection = (sectionName == 'tapis')
                     if inTapisSection:
                        self.configData = {'baseURL':"https://tacc.tapis.io",
                                           'userName':"",
                                           'password':"",
                                           'clientId':"",
                                           'clientKey':"",
                                           'accessToken':"",
                                           'tenantId':"tacc",
                                           'clientName':"",
                                           'clientDescription':""
                                          }
                  elif inTapisSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = list(self.configData[key].keys())[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.configData[key] = {} 
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           sys.stderr.write("Undefined key = value pair %s = %s\n" % (key,value))
               else:
                  eof = True
                  configured = True
         except (IOError,OSError):
            sys.stderr.write("%s could not be read\n" % (configurationFilePath))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         sys.stderr.write("%s could not be opened\n" % (configurationFilePath))

      if configured:
         if   self.authenticationMode == 'accessOnly':
            configurationFilePath = os.path.join(self.configurationDirectory,'accessToken.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['accessToken'] = fpConfig.readline().strip()
            except:
               configured = False
         elif self.authenticationMode == 'clientManager':
            configurationFilePath = os.path.join(self.configurationDirectory,'authenticator.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['password'] = fpConfig.readline().strip()
            except:
               configured = False
         elif self.authenticationMode == 'tokenGenerator':
            configurationFilePath = os.path.join(self.configurationDirectory,'authenticator.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['password'] = fpConfig.readline().strip()
            except:
               configured = False
         elif self.authenticationMode == 'resetToken':
            configurationFilePath = os.path.join(self.configurationDirectory,'authenticator.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['password'] = fpConfig.readline().strip()
            except:
               configured = False

            clientConfigurationPath = os.path.join(self.configurationDirectory,'client.conf')
            try:
               with open(clientConfigurationPath,'r') as fpCache:
                  currentCache = json.load(fpCache)
               self.configData['clientId']  = currentCache['client_id']
               self.configData['clientKey'] = currentCache['client_key']
            except:
               configured = False

      return configured


   def __getAccessOnlySession(self):
      if self.configured:
         super().__init__(base_url=self.configData['baseURL'],
                          tenant_id=self.configData['tenantId'],
                          access_token=self.configData['accessToken'])


   def __getClientManagerSession(self):
      if self.configured:
         super().__init__(base_url=self.configData['baseURL'],
                          tenant_id=self.configData['tenantId'],
                          username=self.configData['userName'],
                          password=self.configData['password'])
         self.get_tokens()


   def __resetToken(self):
      if self.configured:
         super().__init__(base_url=self.configData['baseURL'],
                          username=self.configData['userName'],
                          password=self.configData['password'],
                          client_id=self.configData['clientId'],
                          client_key=self.configData['clientKey'])
         self.get_tokens()
         self.refresh_tokens()
         configurationFilePath = os.path.join(self.configurationDirectory,'accessToken.txt')
         try:
            with open(configurationFilePath,'w') as fpConfig:
               fpConfig.write(self.access_token.access_token)
               fpConfig.write('\n')
         except:
            self.configured = False


   def saveClientConfiguration(self,
                               client):
      clientSaved = False
      if self.configured:
         clientConfigurationPath = os.path.join(self.configurationDirectory,'client.conf')
         try:
            with open(clientConfigurationPath,'w') as fpCache:
               json.dump(client.__dict__,fpCache,
                         indent=3,separators=(',',':'))
               fpCache.write("\n")
            os.chmod(clientConfigurationPath,stat.S_IRUSR|stat.S_IWUSR)
            clientSaved = True
         except:
            pass

      return clientSaved


   @staticmethod
   def decodeJsonToTapis(obj):
      def report(**kwargs):
         for k, v in kwargs.items():
            print(k,type(v))
            try:
               for v2 in v:
                  print(k,type(v2))
            except:
               pass

      objType = obj.pop('__type__',None)
      if   objType == "datetime":
         zone,offset = obj.pop("tz")
         when = obj.pop("when",None)
         obj["tzinfo"] = tzoffset(zone,offset)
         return datetime.datetime(**obj)
      elif objType == "TapisResult":
         return TapisResult(**obj)
      else:
         return obj


   @staticmethod
   def encodeTapisToJson(obj):
      if   isinstance(obj, datetime.datetime):
         encoded = {"__type__": "datetime",
                    "year": obj.year,
                    "month" : obj.month,
                    "day" : obj.day,
                    "hour" : obj.hour,
                    "minute" : obj.minute,
                    "second" : obj.second,
                    "microsecond" : obj.microsecond,
                    "tz": (obj.tzinfo.tzname(obj), obj.utcoffset().total_seconds()),
                    "when": obj.strftime("%c %z")
                   }
      elif isinstance(obj,TapisResult):
         encoded = {"__type__": "TapisResult",**obj.__dict__}
      else:
         raise TypeError("Can't serialize {}".format(obj))

      return encoded


   @staticmethod
   def parseTime(ftime):
      """ Convert timestamp to local time

      Expect format: 2018-07-10T12:28:01.000-05:00
                     2019-05-26 13:32:10-05:00

      """
      # 2018-07-10T12:28:01.000-05:00 (rm last ':')
      if ftime[-1] == 'Z': ftime = ftime[:-1] + 'UTC'

      try:
         ftime = time.strptime(ftime,'%Y-%m-%dT%H:%M:%S%Z')
      except ValueError:
         tz = ftime[-5:] # get the timezone part (i.e., -0500)
         ftime = time.strptime(ftime[:-5],'%Y-%m-%dT%H:%M:%S.%f')
         if tz.startswith("-"):
            sign = -1
            tz = tz[1:] # get the hours offset
         elif tz.startswith("+"):
            tz = tz[1:] # get the hours offset
         seconds = ( int(tz[0:2])*60 + int(tz[2:4]) ) * 60
         seconds *= sign
         # Convert ftime to seconds since epoch,
         ftime = time.mktime(ftime) + seconds
         # Convert seconds since epoch to localtime.
         ftime = time.localtime(ftime)

      outtime = time.strftime("%b %d %H:%M", ftime).split()

      return ftime, outtime


   @staticmethod
   def printFileList(filesList,
                     printKey='name',
                     long_format=False):
      # Sort results alphabetically.
      sortedFilesList = sorted(filesList, key=attrgetter(printKey), reverse=False)

      # Get the length of the longest filename.
      longest_name = 8
      for f in sortedFilesList:
         if len(f.get(printKey)) > longest_name:
            longest_name = len(f.get(printKey))
      # Get the size for the biggest file.
      largest_file = 0
      for f in sortedFilesList:
         if len(str(f.get('size'))) > largest_file:
            largest_file = len(str(f.get('size')))

      # Get size of terminal.
      terminal_size = shutil.get_terminal_size()
      terminal_size_columns = terminal_size.columns 
      columns = terminal_size.columns // longest_name

      if not long_format:
         files = ""
         line_length = 0
         for f in sortedFilesList:
            name = f.get(printKey)
            # If directory, then append '/' to its name.
            if f.get('type') == 'dir':
               name += "/"

            # Give enough space to print name of file/directory.
            name += " " * (longest_name - len(name) + 3)

            # If there is not enough space in the current line to print this
            # file then move to the next line.
            if line_length + len(name) > terminal_size_columns:
               files += "\n"
               line_length = 0

            line_length += len(name)
            files += name

         # Print all files.
         print(files)
      else: # Print file long format.
         for f in sortedFilesList:
            # File permissions and name.
            name   = f.get(printKey)

            perm = f.get('nativePermissions')[0:3]
            # Prefix perfmissions for directories with 'd'.
            if f.get('type') == 'dir':
               perm = "d{}".format(perm)
               name += "/"
            name += " " * (longest_name - len(name) + 3)

            # File size.
            fsize = f.get('size')

            # Date created.
            ftime, outtime = TapisAPI.parseTime( f.get('lastModified') )

            print("{0:<4} {1:>{size_width}} {2:<3} {3:>2} {4:<5} {5:}".format(
                    perm, fsize, outtime[0], ftime.tm_mday, outtime[2], name, 
                    size_width=largest_file))


