#
# @package      hubzero-submit-distributor
# @file         Scripts/tapis2/TapisAPI.py
# @copyright    Copyright 2004-2022 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2022 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import sys
import re
import json
import datetime
from dateutil.tz import tzoffset
from operator import itemgetter
import shutil
import time
import py
from collections import deque
from requests.exceptions import HTTPError

from agavepy.agave import Agave

BASEDIR                = os.path.dirname(os.path.abspath(__file__))
CONFIGURATIONDIRECTORY = os.path.join(BASEDIR,'Configuration')

class TapisAPI(Agave):
   def __init__(self,
                configurationDirectory=CONFIGURATIONDIRECTORY,
                authenticationMode='accessOnly'):
      self.configurationDirectory = configurationDirectory
      self.authenticationMode     = authenticationMode

      self.configData = {}
      self.configured = self.__configure()

      if self.configured:
         if   authenticationMode == 'accessOnly':
            self.__getAccessOnlySession()
         elif authenticationMode == 'clientManager':
            self.__getClientManagerSession()
         elif authenticationMode == 'tokenGenerator':
            self.__getTokenGeneratorSession()
         elif authenticationMode == 'resetToken':
            self.__resetToken()


   def __configure(self):
      configured = False

      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inTapisSection = False

      configurationFilePath = os.path.join(self.configurationDirectory,'tapis.conf')
      try:
         fpConfig = open(configurationFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inTapisSection = (sectionName == 'tapis')
                     if inTapisSection:
                        self.configData = {'userName':"",
                                           'password':"",
                                           'apiKey':"",
                                           'apiSecret':"",
                                           'accessToken':"",
                                           'tenantId':"tacc.prod",
                                           'clientName':"",
                                           'clientDescription':"",
                                           'tapisCachePath':""
                                          }
                  elif inTapisSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = list(self.configData[key].keys())[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.configData[key] = {} 
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           sys.stderr.write("Undefined key = value pair %s = %s\n" % (key,value))
               else:
                  eof = True
                  configured = True
         except (IOError,OSError):
            sys.stderr.write("%s could not be read\n" % (configurationFilePath))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         sys.stderr.write("%s could not be opened\n" % (configurationFilePath))

      if configured:
         if self.configData['tapisCachePath']:
            try:
               self.configData['tapisCachePath'] = os.path.expandvars(self.configData['tapisCachePath'])
            except:
               sys.stderr.write("tapisCachePath: %s is not valid\n" % (self.configData['tapisCachePath']))
               configured = False
            else:
               if not os.path.exists(self.configData['tapisCachePath']):
                  sys.stderr.write("tapisCachePath: %s does not exist\n" % (self.configData['tapisCachePath']))
                  configured = False
               else:
                  os.environ['TAPIS_CACHE_DIR'] = self.configData['tapisCachePath']
         else:
            sys.stderr.write("tapisCachePath must be specified\n")
            configured = False

      if configured:
         if   self.authenticationMode == 'accessOnly':
            configurationFilePath = os.path.join(self.configurationDirectory,'accessToken.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['accessToken'] = fpConfig.readline().strip()
            except:
               configured = False
         elif self.authenticationMode == 'clientManager':
            configurationFilePath = os.path.join(self.configurationDirectory,'authenticator.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['password'] = fpConfig.readline().strip()
            except:
               configured = False
         elif self.authenticationMode == 'resetToken':
            configurationFilePath = os.path.join(self.configurationDirectory,'authenticator.txt')
            try:
               with open(configurationFilePath,'r') as fpConfig:
                  self.configData['password'] = fpConfig.readline().strip()
            except:
               configured = False

            currentCacheFilePath = os.path.join(self.configData['tapisCachePath'],'current')
            try:
               with open(currentCacheFilePath,'r') as fpCache:
                  currentCache = json.load(fpCache)
               self.configData['apiKey']    = currentCache['apikey']
               self.configData['apiSecret'] = currentCache['apisecret']
            except:
               configured = False

      return configured


   def __getAccessOnlySession(self):
      if self.configured:
         Agave.__init__(self,
                        tenant_id=self.configData['tenantId'],
                        token=self.configData['accessToken'])


   def __getClientManagerSession(self):
      if self.configured:
         Agave.__init__(self,
                        tenant_id=self.configData['tenantId'],
                        username=self.configData['userName'],
                        password=self.configData['password'])


   def __resetToken(self):
      if self.configured:
         ag = Agave(api_server='https://api.tacc.utexas.edu',
                    username=self.configData['userName'],
                    password=self.configData['password'],
                    api_key=self.configData['apiKey'],
                    api_secret=self.configData['apiSecret'])
         ag.refresh()
         tokens = ag.tokens()
         configurationFilePath = os.path.join(self.configurationDirectory,'accessToken.txt')
         try:
            with open(configurationFilePath,'w') as fpConfig:
               fpConfig.write(tokens['access_token'])
               fpConfig.write('\n')
         except:
            self.configured = False


   def __getTokenGeneratorSession(self):
      if self.configured:
         ag = Agave.restore()
         tokenString = ag.refresh()
         tokens = ag.tokens()
         configurationFilePath = os.path.join(self.configurationDirectory,'accessToken.txt')
         try:
            with open(configurationFilePath,'w') as fpConfig:
               fpConfig.write(tokens['access_token'])
               fpConfig.write('\n')
         except:
            self.configured = False


   @staticmethod
   def decodeJsonToDatetime(obj):
      if obj.pop('__type__',None) == "datetime":
         zone,offset = obj.pop("tz")
         when = obj.pop("when",None)
         obj["tzinfo"] = tzoffset(zone,offset)
         return datetime.datetime(**obj)
      else:
         return obj


   @staticmethod
   def encodeDatetimeToJson(obj):
      if isinstance(obj, datetime.datetime):
         encoded = {"__type__": "datetime",
                    "year": obj.year,
                    "month" : obj.month,
                    "day" : obj.day,
                    "hour" : obj.hour,
                    "minute" : obj.minute,
                    "second" : obj.second,
                    "microsecond" : obj.microsecond,
                    "tz": (obj.tzinfo.tzname(obj), obj.utcoffset().total_seconds()),
                    "when": obj.strftime("%c %z")
                   }
      else:
         raise TypeError("Can't serialize {}".format(obj))

      return encoded


#
# borrowed from agavepy/files/list.py
#
   file_permissions = {
      "READ"          : "-r--",
      "WRITE"         : "--w-",
      "EXECUTE"       : "---x",
      "READ_WRITE"    : "-rw-",
      "READ_EXECUTE"  : "-r-x",
      "WRITE_EXECUTE" : "--wx",
      "ALL"           : "-rwx",
      "NONE"          : "----"
   }

   @staticmethod
   def parseTime(ftime):
      """ Convert timestamp to local time

      Expect format: 2018-07-10T12:28:01.000-05:00
                     2019-05-26 13:32:10-05:00

      """
      # 2018-07-10T12:28:01.000-05:00 (rm last ':')
      if ftime[-3] == ":": ftime = ftime[:-3] + ftime[-2:]

      try:
         ftime = time.strptime(ftime,'%Y-%m-%dT%H:%M:%S.%f%z')
      except ValueError:
         tz = ftime[-5:] # get the timezone part (i.e., -0500)
         ftime = time.strptime(ftime[:-5],'%Y-%m-%dT%H:%M:%S.%f')
         if tz.startswith("-"):
            sign = -1
            tz = tz[1:] # get the hous offset
         elif tz.startswith("+"):
            tz = tz[1:] # get the hous offset
         seconds = ( int(tz[0:2])*60 + int(tz[2:4]) ) * 60
         seconds *= sign
         # Convert ftime to seconds since epoch,
         ftime = time.mktime(ftime) + seconds
         # Convert seconds since epoch to localtime.
         ftime = time.localtime(ftime)

      outtime = time.strftime("%b %d %H:%M", ftime).split()

      return ftime, outtime


   @staticmethod
   def printFileList(filesList,
                     printKey='name',
                     long_format=False):
      # Sort results alphabetically.
      sortedFilesList = sorted(filesList, key=itemgetter(printKey), reverse=False)

      # Get the length of the longest filename.
      longest_name = 8
      for f in sortedFilesList:
         if len(f[printKey]) > longest_name:
            longest_name = len(f[printKey])
      # Get the size for the biggest file.
      largest_file = 0
      for f in sortedFilesList:
         if len(str(f["length"])) > largest_file:
            largest_file = len(str(f["length"]))

      # Get size of terminal.
      try: # python3 prefered
         terminal_size = shutil.get_terminal_size()
         terminal_size_columns = terminal_size.columns 
         columns = terminal_size.columns // longest_name
      except AttributeError:
         # _rows, columns = os.popen('stty size', 'r').read().split()
         columns = py.io.TerminalWriter().fullwidth
         terminal_size_columns = int(columns)
         columns = int(columns) // longest_name


      if not long_format:
         files = ""
         line_length = 0
         for f in sortedFilesList:
            name = f[printKey]
            # If directory, then append '/' to its name.
            if f["type"] == "dir":
               name += "/"

            # Give enough space to print name of file/directory.
            name += " " * (longest_name - len(name) + 3)

            # If there is not enough space in the current line to print this
            # file then move to the next line.
            if line_length + len(name) > terminal_size_columns:
               files += "\n"
               line_length = 0

            line_length += len(name)
            files += name

         # Print all files.
         print(files)

      else: # Print file long format.
         for f in sortedFilesList:
            # File permissions and name.
            name   = f[printKey]

            # Set file permissions. Two different spellings
            try:
               perm = TapisAPI.file_permissions[f["permissions"]]
            except:
               perm = TapisAPI.file_permissions[f["permission"]]
            # Prefix perfmissions for directories with 'd'.
            if f["type"] == "dir":
               perm = "d{}".format(perm[1:])
               name += "/"
            name += " " * (longest_name - len(name) + 3)

            # File size.
            fsize = f["length"]

            # Date created.
            ftime, outtime = TapisAPI.parseTime( f["lastModified"].strftime("%Y-%m-%dT%H:%M:%S.%f%z") )

            print("{0:<4} {1:>{size_width}} {2:<3} {3:>2} {4:<5} {5:}".format(
                    perm, fsize, outtime[0], ftime.tm_mday, outtime[2], name, 
                    size_width=largest_file))


   def getJobOutputFilesList(self,
                             jobId):
      jobOutputs = []
      try:
         listDirectories = deque(".")
         while len(listDirectories) > 0:
            listDirectory = listDirectories.popleft()
            offset = 0
            allDone = False
            while not allDone:
               partialJobOutputs = self.jobs.listOutputs(jobId=jobId,filePath=listDirectory,offset=offset)
               jobOutputs += partialJobOutputs
               offset += len(partialJobOutputs)
               if len(partialJobOutputs) == 0:
                  allDone = True
               else:
                  for jobOutput in partialJobOutputs:
                     if jobOutput['type'] == 'dir':
                        listDirectories.append(os.path.join(listDirectory,jobOutput['name']))
      except HTTPError as err:
         sys.stderr.write("HTTPError: ")
         sys.stderr.write(err.strerror)
         sys.stderr.write("\n")
         jobOutputs = []
      except Exception as err:
         sys.stderr.write("Exception: ")
         sys.stderr.write(str(err))
         sys.stderr.write("\n")
         jobOutputs = []

      return jobOutputs


