# @package      hubzero-submit-distributor
# @file         VenueMechanismLocal.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import re
import stat
import shutil
import time
import datetime
import logging

from hubzero.submit.LogMessage             import getLogIDMessage as getLogMessage
from hubzero.submit.JobStatistic           import JobStatistic
from hubzero.submit.JobOutput              import JobOutput
from hubzero.submit.VenueMechanismCore     import VenueMechanismCore
from hubzero.submit.RemoteBatchTAPIS2      import RemoteBatchTAPIS2
from hubzero.submit.RemoteBatchTAPIS3      import RemoteBatchTAPIS3
from hubzero.submit.RemoteBatchBOINC       import RemoteBatchBOINC
from hubzero.submit.RemoteBatchCONDOR      import RemoteBatchCONDOR
from hubzero.submit.RemoteBatchFACTORY     import RemoteBatchFACTORY
from hubzero.submit.RemoteBatchPEGASUS     import RemoteBatchPEGASUS
from hubzero.submit.RemoteBatchAppScript   import RemoteBatchAppScript
from hubzero.submit.RemoteBatchLL          import RemoteBatchLL
from hubzero.submit.RemoteBatchLSF         import RemoteBatchLSF
from hubzero.submit.RemoteBatchPBS         import RemoteBatchPBS
from hubzero.submit.RemoteBatchSGE         import RemoteBatchSGE
from hubzero.submit.RemoteBatchSLURM       import RemoteBatchSLURM
from hubzero.submit.RemoteInstantAppScript import RemoteInstantAppScript
from hubzero.submit.RemoteInstantSCRIPT    import RemoteInstantSCRIPT
from hubzero.submit.RemoteWorkflowPEGASUS  import RemoteWorkflowPEGASUS

class VenueMechanismLocal(VenueMechanismCore):
   def __init__(self,
                remoteMonitors,
                hubUserName,
                hubUserId,
                submitterClass,
                currentWorkingDirectory,
                session,
                distributorPid,
                batchCommands,
                isParametric,
                runName,
                localJobId,
                instanceId,
                destination,
                tailFiles,
                enteredCommand,
                gridsite,
                pegasusSiteInfo,
                stageInTarFile,
                auxiliaryTarFile,
                transferExecutable,
                executable,
                appsAccessInfo,
                event,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                disableJobMonitoring,
                siteInfo,
                tapis2SiteInfo,
                tapis3SiteInfo,
                fileMoverInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nCpus,
                nNodes,
                ppn,
                nGpus,
                gpn,
                memoryMB,
                wallTime,
                quotaLimit,
                x509SubmitProxy,
                disableProbeCheck,
                timeHistoryLogs,
                useSetup,
                pegasusVersion,
                pegasusHome):
      VenueMechanismCore.__init__(self,session,hubUserName,hubUserId,submitterClass,distributorPid,
                                       timeHistoryLogs,disableJobMonitoring,
                                       siteInfo,tapis2SiteInfo,tapis3SiteInfo,fileMoverInfo,
                                       toolFilesInfo,dockerImageInfo,submissionScriptsInfo,managerInfo,
                                       remoteMonitors,isMultiCoreRequest,nCpus,nNodes,ppn,nGpus,gpn,memoryMB,event)

      self.logger                           = logging.getLogger(__name__)
      self.jobOutput                        = JobOutput()
      self.venueMechanism                   = 'local'
      if self.fileMoverInfo:
         self.preStageJobInput              = os.path.join(self.fileMoverInfo['remoteBinDirectory'],
                                                           batchCommands['preStageJobInput'])
         self.copyJobInput                  = os.path.join(self.fileMoverInfo['remoteBinDirectory'],
                                                           batchCommands['copyJobInput'])
      else:
         self.preStageJobInput              = ""
         self.copyJobInput                  = ""
      remoteScriptsDirectory = siteInfo['remoteBinDirectory'].replace('$',r'\$')
      self.receiveInput                     = os.path.join(remoteScriptsDirectory,batchCommands['receiveInput'])
      self.submitBatchJob                   = os.path.join(remoteScriptsDirectory,batchCommands['submitBatchJob'])
      self.postProcessJob                   = os.path.join(remoteScriptsDirectory,batchCommands['postProcessJob'])
      self.transmitResults                  = os.path.join(remoteScriptsDirectory,batchCommands['transmitResults'])
      self.cleanupJob                       = os.path.join(remoteScriptsDirectory,batchCommands['cleanupJob'])
      self.killBatchJob                     = os.path.join(remoteScriptsDirectory,batchCommands['killBatchJob'])
      self.isParametric                     = isParametric
      self.runName                          = runName
      self.localJobId                       = localJobId
      if instanceId.startswith('WF;'):
         self.instanceId                    = "0"
         self.nInstances                    = int(instanceId.split(';')[1])
      else:
         self.instanceId                    = instanceId
         self.nInstances                    = 1
      self.remoteBatchSystem                = siteInfo['remoteBatchSystem']
      self.destination                      = destination
      self.venue                            = siteInfo['venue']
      self.tailFiles                        = tailFiles
      self.enteredCommand                   = enteredCommand
      self.gridsite                         = gridsite
      self.pegasusSiteInfo                  = pegasusSiteInfo
      self.siteMonitorDesignator            = siteInfo['siteMonitorDesignator']
      self.remoteUser                       = siteInfo['remoteUser']
      self.logUserRemotely                  = siteInfo['logUserRemotely']
      self.sharedUserSpace                  = siteInfo['sharedUserSpace']
      self.stageFiles                       = siteInfo['stageFiles']
      if stageInTarFile.endswith('.gz'):
         self.stageInTarFile                = stageInTarFile
      else:
         self.stageInTarFile                = stageInTarFile + '.gz'
      if auxiliaryTarFile.endswith('.gz'):
         self.auxiliaryTarFile              = auxiliaryTarFile
      else:
         self.auxiliaryTarFile              = auxiliaryTarFile + '.gz'
      self.transferExecutable               = transferExecutable
      self.executable                       = executable
      self.appsAccessInfo                   = appsAccessInfo
      self.stdinput                         = stdinput
      if isinstance(arguments,list):
         self.arguments                     = ""
         for argument in arguments:
            if ' ' in argument:
               self.arguments              += "%s " % (argument.replace('\\ ',' ').replace(' ',r'\ '))
            else:
               self.arguments              += "%s " % (argument)
      else:
         self.arguments                     = arguments
      self.useEnvironment                   = useEnvironment
      self.environment                      = environment
      self.remoteBatchQueue                 = siteInfo['remoteBatchQueue']
      self.pegasusTemplates                 = siteInfo['pegasusTemplates']
      self.executableClassificationsAllowed = siteInfo['executableClassificationsAllowed']
      self.wallTime                         = wallTime
      self.quotaLimit                       = quotaLimit
      self.useSetup                         = useSetup
      self.pegasusVersion                   = pegasusVersion
      self.pegasusHome                      = pegasusHome
      self.x509SubmitProxy                  = x509SubmitProxy
      self.disableProbeCheck                = disableProbeCheck
      self.batchLogPath                     = ""
      self.appScriptPath                    = ""
      self.batchScriptPath                  = ""
      self.nodeFilePath                     = ""
      self.toolInputTemplateFilePath        = ""
      self.toolOutputTemplateFilePath       = ""
      self.remoteJobIdNumber                = ""
      self.sshIdentityPath                  = ""
      self.removeIdentity                   = False
      self.executionMode                    = siteInfo['executionMode']

      if self.remoteBatchSystem == 'SCRIPT':
         self.isBatchJob           = False
         self.instanceDirectory    = currentWorkingDirectory
      else:
         self.isBatchJob           = True
         if self.instanceId != "0":
            self.instanceDirectory = os.path.join(currentWorkingDirectory,self.runName,self.instanceId)
         else:
            self.instanceDirectory = os.path.join(currentWorkingDirectory,self.runName)
         if not os.path.isdir(self.instanceDirectory):
            os.makedirs(self.instanceDirectory)

      self.createIdentityPaths()

      self.workingDirectory        = currentWorkingDirectory
      if siteInfo['remoteScratchDirectory']:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         self.scratchDirectory     = os.path.join(siteInfo['remoteScratchDirectory'],"%s_%s" % (str(epoch),localJobId))
         if self.instanceId != "0":
            self.scratchDirectory += "_%s" % (self.instanceId)
      else:
         self.scratchDirectory     = self.instanceDirectory


   def createIdentityPaths(self):
      if not self.sshIdentityPath:
         if   self.remoteUser.startswith('USER:'):
            self.remoteUser = self.hubUserName
            identityPaths = self.remoteIdentityManager.queryUserIdentities(self.siteInfo['identityManagers'],
                                                                           self.hubUserName)
            if 'personalPKI' in identityPaths:
               self.sshIdentityPath = identityPaths['personalPKI']
               self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
            if 'x509' in identityPaths:
               if not self.x509SubmitProxy:
                  self.x509SubmitProxy = identityPaths['x509']
         elif self.remoteUser.startswith('USER'):
            self.remoteUser = self.hubUserName
            identityPaths = self.remoteIdentityManager.queryUserIdentities(self.siteInfo['identityManagers'],
                                                                           self.hubUserName)
            if 'communitySSH' in identityPaths:
               self.sshIdentityPath = identityPaths['communitySSH']
               self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
            if 'x509' in identityPaths:
               if not self.x509SubmitProxy:
                  self.x509SubmitProxy = identityPaths['x509']
                  self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.x509SubmitProxy))
         else:
            identityPaths = self.remoteIdentityManager.queryUserIdentities(self.siteInfo['identityManagers'],
                                                                           self.hubUserName)
            if 'communitySSH' in identityPaths:
               self.sshIdentityPath = identityPaths['communitySSH']
               self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.sshIdentityPath))
            if 'x509' in identityPaths:
               if not self.x509SubmitProxy:
                  self.x509SubmitProxy = identityPaths['x509']
                  self.logger.log(logging.INFO,getLogMessage("IDENTITY = " + self.x509SubmitProxy))


   def createScripts(self):
      exitCode    = 0
      scriptFiles = []

      self.logger.log(logging.INFO,getLogMessage("workingDirectory " + self.workingDirectory))
      if   self.remoteBatchSystem == 'PBS':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchPBS(self.hubUserName,self.hubUserId,self.submitterClass,
                                           self.session,self.instanceToken,self.wsJobId,
                                           self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                           self.transferExecutable,
                                           self.executable,self.arguments,self.stdinput,
                                           self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                           self.submissionScriptsInfo,
                                           self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,self.wallTime,
                                           self.quotaLimit,
                                           self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'SGE':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchSGE(self.hubUserName,self.hubUserId,self.submitterClass,
                                           self.session,self.instanceToken,self.wsJobId,
                                           self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                           self.transferExecutable,
                                           self.executable,self.arguments,self.stdinput,
                                           self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                           self.submissionScriptsInfo,
                                           self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,self.wallTime,
                                           self.quotaLimit,
                                           self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'LSF':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchLSF(self.hubUserName,self.hubUserId,self.submitterClass,
                                           self.session,self.instanceToken,self.wsJobId,
                                           self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                           self.transferExecutable,
                                           self.executable,self.arguments,self.stdinput,
                                           self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                           self.submissionScriptsInfo,
                                           self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,
                                           self.wallTime,self.quotaLimit,
                                           self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'LL':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchLL(self.hubUserName,self.hubUserId,self.submitterClass,
                                          self.session,self.instanceToken,self.wsJobId,
                                          self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                          self.transferExecutable,
                                          self.executable,self.arguments,self.stdinput,
                                          self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                          self.submissionScriptsInfo,
                                          self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,
                                          self.wallTime,self.quotaLimit,
                                          self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'SLURM':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchSLURM(self.hubUserName,self.hubUserId,self.submitterClass,
                                             self.session,self.instanceToken,self.wsJobId,
                                             self.runName,self.localJobId,self.instanceId,appScriptName,self.environment,
                                             self.transferExecutable,
                                             self.executable,self.arguments,self.stdinput,
                                             self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                             self.submissionScriptsInfo,
                                             self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,
                                             self.wallTime,self.quotaLimit,
                                             self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'CONDOR':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchCONDOR(self.hubUserName,self.hubUserId,self.submitterClass,
                                              self.session,self.instanceToken,self.wsJobId,
                                              self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                              appScriptName,self.environment,
                                              self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                              self.submissionScriptsInfo,
                                              self.managerInfo,self.nGpus,self.gpn,self.memoryMB,self.wallTime)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'BOINC':
         if self.isParametric:
            if self.instanceId == "0":
               appScriptName          = ""
               appScript              = ""
               self.remoteBatch = RemoteBatchBOINC(self.hubUserName,self.hubUserId,self.submitterClass,
                                                   self.session,self.instanceToken,self.wsJobId,
                                                   self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                   appScriptName,self.environment,
                                                   self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                   self.submissionScriptsInfo,
                                                   self.managerInfo,self.wallTime)
               batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
               batchLogName                                      = self.remoteBatch.getBatchLog()
               nodeFileName                                      = ""
               nodeList                                          = []
               toolInputTemplateFileName                         = ""
               toolInputTemplate                                 = ""
               toolOutputTemplateFileName                        = ""
               toolOutputTemplate                                = ""
            else:
               self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                           self.session,self.instanceToken,self.wsJobId,
                                                           self.runName,self.localJobId,self.instanceId,False,
                                                           self.transferExecutable,self.executable,
                                                           self.stdinput,self.arguments,
                                                           self.useEnvironment,self.environment,
                                                           self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                           self.managerInfo,self.timeHistoryLogs)
               appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
               self.remoteBatch = RemoteBatchBOINC(self.hubUserName,self.hubUserId,self.submitterClass,
                                                   self.session,self.instanceToken,self.wsJobId,
                                                   self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                   appScriptName,self.environment,
                                                   self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                   self.submissionScriptsInfo,
                                                   self.managerInfo,self.wallTime)
               batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
               batchLogName                                      = self.remoteBatch.getBatchLog()
               nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
               toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
               toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
         else:
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.session,self.instanceToken,self.wsJobId,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            self.remoteBatch = RemoteBatchBOINC(self.hubUserName,self.hubUserId,self.submitterClass,
                                                self.session,self.instanceToken,self.wsJobId,
                                                self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                                appScriptName,self.environment,
                                                self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                self.submissionScriptsInfo,
                                                self.managerInfo,self.wallTime)
            batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
            batchLogName                                      = self.remoteBatch.getBatchLog()
            nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'FACTORY':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteBatchFACTORY(self.hubUserName,self.hubUserId,self.submitterClass,
                                               self.session,self.instanceToken,self.wsJobId,
                                               self.runName,self.localJobId,self.instanceId,self.instanceDirectory,
                                               appScriptName,self.environment,
                                               self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                               self.submissionScriptsInfo,
                                               self.managerInfo,self.x509SubmitProxy,
                                               self.nGpus,self.gpn,self.memoryMB,self.wallTime)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'PEGASUS':
         if self.isParametric:
            if self.instanceId != "0":
               self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                           self.session,self.instanceToken,self.wsJobId,
                                                           self.runName,self.localJobId,self.instanceId,True,
                                                           self.transferExecutable,self.executable,
                                                           self.stdinput,self.arguments,
                                                           self.useEnvironment,self.environment,
                                                           self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                           self.managerInfo,self.timeHistoryLogs)
               appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
               batchLogName               = ""
               batchScriptName            = ""
               batchScript                = ""
               nodeFileName               = ""
               nodeList                   = []
               toolInputTemplateFileName  = ""
               toolInputTemplate          = ""
               toolOutputTemplateFileName = ""
               toolOutputTemplate         = ""
            else:
               appScriptName = ""
               appScript     = ""
               remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
               self.remoteBatch = RemoteWorkflowPEGASUS(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.session,self.instanceToken,self.wsJobId,
                                                        self.runName,self.localJobId,self.instanceId,
                                                        self.instanceDirectory,remoteScratchDirectory,
                                                        self.useSetup,self.pegasusVersion,self.pegasusHome,
                                                        self.pegasusTemplates,self.x509SubmitProxy,self.sshIdentityPath,
                                                        self.gridsite,self.pegasusSiteInfo,
                                                        self.nGpus,self.gpn,self.memoryMB,self.wallTime,
                                                        self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
               batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.buildWorkflowScript()
               batchLogName               = ""
               nodeFileName               = ""
               nodeList                   = []
               toolInputTemplateFileName  = ""
               toolInputTemplate          = ""
               toolOutputTemplateFileName = ""
               toolOutputTemplate         = ""
         else:
            self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                        self.session,self.instanceToken,self.wsJobId,
                                                        self.runName,self.localJobId,self.instanceId,False,
                                                        self.transferExecutable,self.executable,
                                                        self.stdinput,self.arguments,
                                                        self.useEnvironment,self.environment,
                                                        self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                        self.managerInfo,self.timeHistoryLogs)
            appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
            remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
            self.remoteBatch = RemoteBatchPEGASUS(self.hubUserName,self.hubUserId,self.submitterClass,
                                                  self.session,self.instanceToken,self.wsJobId,
                                                  self.runName,self.localJobId,self.instanceId,
                                                  self.instanceDirectory,remoteScratchDirectory,
                                                  self.useSetup,self.pegasusVersion,self.pegasusHome,
                                                  appScriptName,
                                                  self.executable,self.arguments,self.isMultiCoreRequest,
                                                  self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                  self.submissionScriptsInfo,self.managerInfo,
                                                  self.x509SubmitProxy,self.sshIdentityPath,self.gridsite,self.pegasusSiteInfo,
                                                  self.nGpus,self.gpn,self.memoryMB,self.wallTime,
                                                  self.timeHistoryLogs)
            batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
            batchLogName                                      = self.remoteBatch.getBatchLog()
            nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
            toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
            toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
            userDaxExecutables                                = self.remoteBatch.getUserDaxExecutables()
            for userDaxExecutable in userDaxExecutables:
               submissionAllowed,executableClassification = self.appsAccessInfo.isSubmissionAllowed(userDaxExecutable)
               if not submissionAllowed:
                  message = "Access to %s denied for %s" % (userDaxExecutable,self.hubUserName)
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self._writeToStderr(message + "\n")
                  if not exitCode:
                     exitCode = 9
               if (not '*' in self.executableClassificationsAllowed) and \
                  (not executableClassification in self.executableClassificationsAllowed):
                  message = "The selected site does not meet the specified\n" + \
                            "application access requirements.  Please select another site."
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self._writeToStderr(message + "\n")
                  if not exitCode:
                     exitCode = 10
      elif self.remoteBatchSystem == 'TAPIS2':
         appScriptName       = ""
         appScript           = ""
         appScriptExecutable = False
         self.remoteBatch = RemoteBatchTAPIS2(self.session,self.instanceToken,self.wsJobId,
                                              self.runName,self.localJobId,self.instanceId,
                                              self.tapis2SiteInfo,self.submissionScriptsInfo,
                                              self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,self.wallTime)
         batchScriptName,batchScript = self.remoteBatch.getBatchTapisFile()
         batchScriptExecutable       = False
         batchLogName                = ""
         nodeFileName                = ""
         nodeList                    = []
         toolInputTemplateFileName   = ""
         toolInputTemplate           = ""
         toolOutputTemplateFileName  = ""
         toolOutputTemplate          = ""
      elif self.remoteBatchSystem == 'TAPIS3':
         appScriptName       = ""
         appScript           = ""
         appScriptExecutable = False
         self.remoteBatch = RemoteBatchTAPIS3(self.session,self.instanceToken,self.wsJobId,
                                              self.runName,self.localJobId,self.instanceId,
                                              self.tapis3SiteInfo,self.submissionScriptsInfo,
                                              self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,self.wallTime)
         batchScriptName,batchScript = self.remoteBatch.getBatchTapisFile()
         batchScriptExecutable       = False
         batchLogName                = ""
         nodeFileName                = ""
         nodeList                    = []
         toolInputTemplateFileName   = ""
         toolInputTemplate           = ""
         toolOutputTemplateFileName  = ""
         toolOutputTemplate          = ""
      elif self.remoteBatchSystem == 'SCRIPT':
         self.remoteAppScript = RemoteInstantAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                       self.session,self.instanceToken,self.wsJobId,
                                                       self.runName,self.localJobId,self.instanceId,False,
                                                       self.workingDirectory,
                                                       self.transferExecutable,self.executable,
                                                       self.stdinput,self.arguments,
                                                       self.useEnvironment,self.environment,
                                                       self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                       self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,self.submitterClass,
                                                self.session,self.instanceToken,self.wsJobId,
                                                self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                                self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                self.submissionScriptsInfo,
                                                self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,
                                                self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()
      elif self.remoteBatchSystem == 'BGSCRIPT':
         self.remoteAppScript = RemoteBatchAppScript(self.hubUserName,self.hubUserId,self.submitterClass,
                                                     self.session,self.instanceToken,self.wsJobId,
                                                     self.runName,self.localJobId,self.instanceId,False,
                                                     self.transferExecutable,self.executable,
                                                     self.stdinput,self.arguments,
                                                     self.useEnvironment,self.environment,
                                                     self.isMultiCoreRequest,self.siteInfo,self.submissionScriptsInfo,
                                                     self.managerInfo,self.timeHistoryLogs)
         appScriptName,appScript,appScriptExecutable = self.remoteAppScript.buildAppScript()
         self.remoteBatch = RemoteInstantSCRIPT(self.hubUserName,self.hubUserId,self.submitterClass,
                                                self.session,self.instanceToken,self.wsJobId,
                                                self.runName,self.localJobId,self.instanceId,self.workingDirectory,appScriptName,
                                                self.isMultiCoreRequest,self.siteInfo,self.toolFilesInfo,self.dockerImageInfo,
                                                self.submissionScriptsInfo,
                                                self.managerInfo,self.nNodes,self.ppn,self.nGpus,self.gpn,self.memoryMB,
                                                self.timeHistoryLogs)
         batchScriptName,batchScript,batchScriptExecutable = self.remoteBatch.getBatchScript()
         batchLogName                                      = self.remoteBatch.getBatchLog()
         nodeFileName,nodeList                             = self.remoteBatch.getBatchNodeList()
         toolInputTemplateFileName,toolInputTemplate       = self.remoteBatch.getBatchToolInputTemplate()
         toolOutputTemplateFileName,toolOutputTemplate     = self.remoteBatch.getBatchToolOutputTemplate()

      if batchLogName != "":
         self.batchLogPath               = os.path.join(self.instanceDirectory,batchLogName)
      if appScriptName != "":
         self.appScriptPath              = os.path.join(self.instanceDirectory,appScriptName)
      if batchScriptName != "":
         self.batchScriptPath            = os.path.join(self.instanceDirectory,batchScriptName)
      if nodeFileName != "":
         self.nodeFilePath               = os.path.join(self.instanceDirectory,nodeFileName)
      if toolInputTemplateFileName != "":
         self.toolInputTemplateFilePath  = os.path.join(self.instanceDirectory,toolInputTemplateFileName)
      if toolOutputTemplateFileName != "":
         self.toolOutputTemplateFilePath = os.path.join(self.instanceDirectory,toolOutputTemplateFileName)

      jobSubmissionMechanism = self.venueMechanism + self.remoteBatchSystem
      self.jobIndex = int(self.instanceId)
      if not self.jobIndex in self.jobStatistics:
         self.jobStatistics[self.jobIndex] = JobStatistic(self.nCpus)
      self.jobStatistics[self.jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism
      if self.event:
         self.jobStatistics[self.jobIndex]['event'] = self.event

      if batchScript != "":
         try:
            fpBatchScript = open(self.batchScriptPath,'w')
            try:
               fpBatchScript.write(batchScript)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.batchScriptPath)))
               self._writeToStderr("%s could not be written\n" % (self.batchScriptPath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
            else:
               scriptFiles.append(self.batchScriptPath)
               if batchScriptExecutable:
                  os.chmod(self.batchScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            finally:
               fpBatchScript.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.batchScriptPath)))
            self._writeToStderr("%s could not be opened\n" % (self.batchScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if appScript != "":
         try:
            fpAppScript = open(self.appScriptPath,'w')
            try:
               fpAppScript.write(appScript)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.appScriptPath)))
               self._writeToStderr("%s could not be written\n" % (self.appScriptPath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
            else:
               scriptFiles.append(self.appScriptPath)
               if appScriptExecutable:
                  os.chmod(self.appScriptPath,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            finally:
               fpAppScript.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.appScriptPath)))
            self._writeToStderr("%s could not be opened\n" % (self.appScriptPath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if len(nodeList) > 0:
         try:
            fpNodes = open(self.nodeFilePath,'w')
            try:
               fpNodes.write('\n'.join(nodeList)+'\n')
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.nodeFilePath)))
               self._writeToStderr("%s could not be written\n" % (self.nodeFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
            else:
               scriptFiles.append(self.nodeFilePath)
            finally:
               fpNodes.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.nodeFilePath)))
            self._writeToStderr("%s could not be opened\n" % (self.nodeFilePath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if toolInputTemplate != "":
         try:
            fpToolInputTemplate = open(self.toolInputTemplateFilePath,'w')
            try:
               fpToolInputTemplate.write(toolInputTemplate)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.toolInputTemplateFilePath)))
               self._writeToStderr("%s could not be written\n" % (self.toolInputTemplateFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
            finally:
               fpToolInputTemplate.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.toolInputTemplateFilePath)))
            self._writeToStderr("%s could not be opened\n" % (self.toolInputTemplateFilePath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if toolOutputTemplate != "":
         try:
            fpToolOutputTemplate = open(self.toolOutputTemplateFilePath,'w')
            try:
               fpToolOutputTemplate.write(toolOutputTemplate)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.toolOutputTemplateFilePath)))
               self._writeToStderr("%s could not be written\n" % (self.toolOutputTemplateFilePath))
               self.jobStatistics[0]['exitCode'] = 1
               exitCode = 1
            finally:
               fpToolOutputTemplate.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.toolOutputTemplateFilePath)))
            self._writeToStderr("%s could not be opened\n" % (self.toolOutputTemplateFilePath))
            self.jobStatistics[0]['exitCode'] = 1
            exitCode = 1

      if not exitCode:
         self.scriptsCreated = True

      return(exitCode,scriptFiles)


   def sendFiles(self):
      self.filesSent = True

      exitStatus = 0
      self.jobStatistics[self.jobIndex]['venue'] = self.venue

      remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
      command = self.receiveInput + " " + self.instanceDirectory + " " + \
                                          remoteScratchDirectory + " " + \
                                          os.path.join(self.instanceDirectory,self.timeHistoryLogs['timestampTransferred'])
      self.logger.log(logging.INFO,getLogMessage("command = " + command))
      exitStatus,stdOutput,stdError = self.executeCommand(command)
      self.logger.log(logging.INFO,getLogMessage(stdOutput))

      if exitStatus:
         self.filesSent = False
         self.jobStatistics[self.jobIndex]['exitCode'] = 11
#        if stdOutput != "":
#           if self.instanceId != "0":
#              stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#           else:
#              stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#           try:
#              fpStd = open(stdFile,'a')
#              try:
#                 fpStd.write(stdOutput)
#              except (IOError,OSError):
#                 self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#              finally:
#                 fpStd.close()
#           except (IOError,OSError):
#              self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         if stdError != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdError)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
         self.removeJobRegistration()

      return(self.filesSent)


   def executeJob(self):
      exitStatus = 0

      if self.remoteBatchSystem == 'SCRIPT':
         if self.batchScriptPath != "":
            command = self.batchScriptPath
         else:
            command = self.appScriptPath
         self.logger.log(logging.INFO,getLogMessage("command = " + command))

         self.removeJobRegistration()

         exitStatus,stdOutput,stdError = self.executeCommand(command,True)
         self.jobSubmitted = True
         self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
      else:
         commandArgs = []
         remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
         commandArgs.append(self.submitBatchJob)
         commandArgs.append(self.instanceDirectory)
         commandArgs.append(remoteScratchDirectory)
         commandArgs.append(os.path.join('.',os.path.basename(self.batchScriptPath)))
         if self.disableJobMonitoring:
            commandArgs.append('1')
         if self.logUserRemotely:
            hubUserHash = self.remoteIdentityManager.queryUserHash(self.siteInfo['identityManagers'],
                                                                   self.hubUserName,
                                                                   self.hubUserId)
            commandArgs.append(hubUserHash)
            commandArgs.append(str(self.hubUserId))
            commandArgs.append(self.localJobId.lstrip('0') + '_' + self.instanceId)
            commandArgs.append(self.executable)
         command = ' '.join(commandArgs)
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         exitStatus,stdOutput,stdError = self.executeLaunchCommand(command)

         if not exitStatus:
            remoteJobId = stdOutput.strip()
            self.logger.log(logging.INFO,getLogMessage("remoteJobId = " + remoteJobId))
            self.remoteJobIdNumber = self.remoteBatch.getRemoteJobIdNumber(remoteJobId)
            if self.remoteJobIdNumber != '-1':
               self.jobStatistics[self.jobIndex]['remoteJobIdNumber'] = self.remoteJobIdNumber
            else:
               exitStatus = 248

            if not exitStatus:
               self.jobSubmitted = True
               if   self.remoteBatchSystem == 'PEGASUS':
                  self.remoteJobMonitor.postPegasusWorkflowSubmission(self.session,self.instanceToken,self.wsJobId,
                                                                      self.siteInfo['siteName'],
                                                                      self.siteInfo['identityManagers'],
                                                                      self.siteMonitorDesignator,
                                                                      self.remoteJobIdNumber,str(self.hubUserId),
                                                                      self.enteredCommand,
                                                                      self.localJobId,self.nInstances,self.destination,
                                                                      self.runName,self.nCpus,self.distributorPid)
               elif self.remoteBatchSystem == 'BOINC' and self.instanceId == "0":
                  self.remoteJobMonitor.postBoincWorkflowSubmission(self.session,self.instanceToken,self.wsJobId,
                                                                    self.siteInfo['siteName'],
                                                                    self.siteInfo['identityManagers'],
                                                                    self.siteMonitorDesignator,
                                                                    self.remoteJobIdNumber,str(self.hubUserId),
                                                                    self.enteredCommand,
                                                                    self.localJobId,self.nInstances,
                                                                    self.instancesInitialJobStatus,self.destination,
                                                                    self.runName,self.nCpus,self.distributorPid)
               else:
                  self.remoteJobMonitor.postJobSubmission(self.session,self.instanceToken,self.wsJobId,
                                                          self.siteInfo['siteName'],
                                                          self.siteInfo['identityManagers'],
                                                          self.siteMonitorDesignator,
                                                          self.remoteJobIdNumber,str(self.hubUserId),
                                                          self.tailFiles,self.enteredCommand,
                                                          self.scratchDirectory,
                                                          self.localJobId,self.instanceId,self.destination,
                                                          self.runName,self.nCpus,self.distributorPid)
            else:
               self.removeJobRegistration()
               self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus
         else:
            self.removeJobRegistration()
            self.jobStatistics[self.jobIndex]['exitCode'] = exitStatus

      if not self.jobSubmitted:
         if stdOutput != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdOutput)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         if stdError != "":
            if self.instanceId != "0":
               stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
            else:
               stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
            try:
               fpStd = open(stdFile,'a')
               try:
                  fpStd.write(stdError)
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
               finally:
                  fpStd.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

      return(self.jobSubmitted)


   def getWaitForJobInfo(self):
      waitForJobInfo = {}
      waitForJobInfo['isBatchJob']            = self.isBatchJob
      waitForJobInfo['siteMonitorDesignator'] = self.siteMonitorDesignator
      waitForJobInfo['remoteJobId']           = self.remoteJobIdNumber
      waitForJobInfo['knownSite']             = self.destination
      waitForJobInfo['executionMode']         = self.executionMode
      waitForJobInfo['instanceToken']         = self.instanceToken

      return(waitForJobInfo)


   def postProcess(self):
      if not self.jobPostProcessed:
         if self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and self.instanceId == "0") or not self.isParametric:
                  if not self.sharedUserSpace:
                     remoteWorkingDirectory = self.workingDirectory.replace('$',r'\$')
                  else:
                     remoteWorkingDirectory = self.instanceDirectory
                  remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
                  command = self.postProcessJob + " " + remoteWorkingDirectory + " " + \
                                                        remoteScratchDirectory + " " + \
                                                        "PEGASUS"
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  exitStatus,stdOutput,stdError = self.executeCommand(command)
                  self.logger.log(logging.INFO,getLogMessage(stdOutput))
                  if exitStatus:
                     self.logger.log(logging.INFO,getLogMessage(stdError))

                  self.jobPostProcessed = True


   def retrieveFiles(self):
      exitStatus = 0

      if self.sharedUserSpace and self.stageFiles:
         outTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
         stageOutTarFiles = []
         stageOutTarFiles.append(os.path.join(self.instanceDirectory,"%s.gz" % (outTarFile)))
         stageOutTarFiles.append(os.path.join(self.instanceDirectory,outTarFile))
         stageOutTarFiles.append("%s.gz" % (outTarFile))
         stageOutTarFiles.append(outTarFile)
         stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),"%s.gz" % (outTarFile)))
         stageOutTarFiles.append(os.path.join(os.path.dirname(self.instanceDirectory),outTarFile))

         fetchResults = True
         for stageOutTarFile in stageOutTarFiles:
            if os.path.isfile(stageOutTarFile):
               fetchResults = False
         if fetchResults and self.jobSubmitted:
            stageOutTarFile = "%s_%s_output.tar" % (self.localJobId,self.instanceId)
            command = self.transmitResults + " " + self.instanceDirectory + " " + \
                                                   stageOutTarFile
            self.logger.log(logging.INFO,getLogMessage("command = " + command))
            exitStatus,stdOutput,stdError = self.executeCommand(command)
            self.logger.log(logging.INFO,getLogMessage(stdOutput))
            if exitStatus:
               self.jobStatistics[self.jobIndex]['exitCode'] = 12
#              if stdOutput != "":
#                 if self.instanceId != "0":
#                    stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
#                 else:
#                    stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
#                 try:
#                    fpStd = open(stdFile,'a')
#                    try:
#                       fpStd.write(stdOutput)
#                    except (IOError,OSError):
#                       self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
#                    finally:
#                       fpStd.close()
#                 except (IOError,OSError):
#                    self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

               if stdError != "":
                  if self.instanceId != "0":
                     stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                  else:
                     stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                  try:
                     fpStd = open(stdFile,'a')
                     try:
                        fpStd.write(stdError)
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                     finally:
                        fpStd.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

         for stageOutTarFile in stageOutTarFiles:
            if os.path.isfile(stageOutTarFile):
               if stageOutTarFile.endswith('.gz'):
                  command = "tar xzmf " + stageOutTarFile + \
                                      " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
               else:
                  command = "tar xmf " + stageOutTarFile + \
                                     " --ignore-case --exclude '*hub-proxy.*' -C " + self.instanceDirectory
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               exitStatus,stdOutput,stdError = self.executeCommand(command)
               if exitStatus == 0:
                  try:
                     os.remove(stageOutTarFile)
                  except:
                     pass
               else:
                  self.jobStatistics[self.jobIndex]['exitCode'] = 12
                  if stdOutput != "":
                     if self.instanceId != "0":
                        stdFile = os.path.join(self.instanceDirectory,"%s_%s.stdout" % (self.runName,self.instanceId))
                     else:
                        stdFile = os.path.join(self.instanceDirectory,"%s.stdout" % (self.runName))
                     try:
                        fpStd = open(stdFile,'a')
                        try:
                           fpStd.write(stdOutput)
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                        finally:
                           fpStd.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))

                  if stdError != "":
                     if self.instanceId != "0":
                        stdFile = os.path.join(self.instanceDirectory,"%s_%s.stderr" % (self.runName,self.instanceId))
                     else:
                        stdFile = os.path.join(self.instanceDirectory,"%s.stderr" % (self.runName))
                     try:
                        fpStd = open(stdFile,'a')
                        try:
                           fpStd.write(stdError)
                        except (IOError,OSError):
                           self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (stdFile)))
                        finally:
                           fpStd.close()
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (stdFile)))
               break

      if not exitStatus:
         self.filesRetrieved = True


   def cleanupFiles(self):
      if not self.filesCleanedup:
         if   self.remoteBatchSystem == 'PEGASUS':
            if self.jobSubmitted:
               if (self.isParametric and self.instanceId == "0") or not self.isParametric:
                  remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
                  if self.isParametric:
                     self.jobOutput.getPegasusStdTimeFiles(self.instanceDirectory,
                                                           remoteScratchDirectory,
                                                           self.timeHistoryLogs['timeResults'])
                  else:
                     self.jobOutput.processPegasusFiles(self.instanceDirectory,
                                                        remoteScratchDirectory,
                                                        self.timeHistoryLogs['timeResults'])

                  command = self.cleanupJob + " " + self.instanceDirectory + " " + \
                                                    remoteScratchDirectory + " " + \
                                                    self.remoteBatchSystem
                  self.logger.log(logging.INFO,getLogMessage("command = " + command))
                  self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

            workDirectoryName = os.path.join(self.instanceDirectory,'work')
            if os.path.isdir(workDirectoryName):
               shutil.rmtree(workDirectoryName,True)
            inProcessDirectoryName = os.path.join(self.instanceDirectory,'InProcessResults')
            if os.path.isdir(inProcessDirectoryName):
               shutil.rmtree(inProcessDirectoryName,True)

            self.cleanupPegasusFiles()
         elif self.remoteBatchSystem == 'BOINC':
            if self.jobSubmitted:
               if self.instanceId != "0":
                  timestampTransferredFile = self.timeHistoryLogs['timestampTransferred']
                  batchTransferredTimeFile = timestampTransferredFile.replace('_'+self.instanceId,'_0')
                  batchTransferredTimePath = os.path.join(self.instanceDirectory,batchTransferredTimeFile)
                  if os.path.exists(batchTransferredTimePath):
                     shutil.copy2(batchTransferredTimePath,os.path.join(self.instanceDirectory,timestampTransferredFile))

               remoteScratchDirectory = self.scratchDirectory.replace('$',r'\$')
               command = self.cleanupJob + " " + self.instanceDirectory + " " + \
                                                 remoteScratchDirectory + " " + \
                                                 self.remoteBatchSystem
               self.logger.log(logging.INFO,getLogMessage("command = " + command))
               self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

         if self.removeIdentity:
            if os.path.isfile(self.sshIdentityPath):
               try:
                  os.remove(self.sshIdentityPath)
               except:
                  pass

         self.cleanupStageInTarFile()
         self.cleanupScriptTemplateLogFiles()

         if self.sharedUserSpace and self.stageFiles:
            self.cleanupStageOutTarFile()

         self.filesCleanedup = True


   def killScripts(self,
                   signalNumber):
      if self.remoteJobIdNumber:
         command = self.killBatchJob + " " + self.remoteJobIdNumber + " " + self.remoteBatchSystem
         self.logger.log(logging.INFO,getLogMessage("command = " + command))
         self.logger.log(logging.INFO,getLogMessage(self.executeCommand(command)[1]))

      maximumJobIndex = max(self.jobStatistics.keys())
      self.jobStatistics[maximumJobIndex]['exitCode'] = 1 << 7 | signalNumber

      self.scriptsKilled = True


