#!/usr/bin/env python3
#
# @package      hubzero-submit-distributor
# @file         consistencyCheck.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2025 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import csv
import copy
import logging

# Check for missing filemovers

from hubzero.submit.ApplicationUtilities import ApplicationUtilities as ApplicationUtilities
from hubzero.submit.InfosInfo            import InfosInfo
from hubzero.submit.SitesInfo            import SitesInfo
from hubzero.submit.Tapis2SitesInfo      import Tapis2SitesInfo
from hubzero.submit.Tapis3SitesInfo      import Tapis3SitesInfo
from hubzero.submit.ToolsInfo            import ToolsInfo
from hubzero.submit.ManagersInfo         import ManagersInfo
from hubzero.submit.ToolFilesInfo        import ToolFilesInfo
from hubzero.submit.DockerImagesInfo     import DockerImagesInfo
from hubzero.submit.AggregatorsInfo      import AggregatorsInfo
from hubzero.submit.IdentitiesInfo       import IdentitiesInfo
from hubzero.submit.MonitorsInfo         import MonitorsInfo

ALLOWEDVENUEMECHANISMS = ['local', 'ssh', 'tapis2', 'tapis3']
PEGASUSVERSION         = "5.0.1"

if __name__ == '__main__':
   applicationUtilities = ApplicationUtilities("DEBUG")
   applicationUtilities.openStreamLogger(sys.stderr,
                                         "DEBUG")

   configurationDirectory = os.path.join(os.sep,'etc','submit')
   infosConfigurationFile = 'infos.conf'
   configFilePath = os.path.join(configurationDirectory,infosConfigurationFile)
   infosInfo = InfosInfo(configFilePath)

   sitesInfo = SitesInfo(infosInfo.getInfoPath('sites'),
                         restrictionUser=None,
                         pegasusTemplateDirectory=infosInfo.getInfoPath('pegasusTemplates'),
                         allowedVenueMechanisms=ALLOWEDVENUEMECHANISMS,
                         pegasusVersion=PEGASUSVERSION)
   tapis2SitesInfo = Tapis2SitesInfo(infosInfo.getInfoPath('tapis2sites'))
   tapis3SitesInfo = Tapis3SitesInfo(infosInfo.getInfoPath('tapis3sites'))

   toolsInfo = ToolsInfo(infosInfo.getInfoPath('tools'))

   managersInfo = ManagersInfo(infosInfo.getInfoPath('managers'))
   usedRemoteManagers = []

   toolFilesInfo = ToolFilesInfo(infosInfo.getInfoPath('toolfiles'))

   dockerImagesInfo = DockerImagesInfo(infosInfo.getInfoPath('dockerimages'))

   aggregatorsInfo = AggregatorsInfo(infosInfo.getInfoPath('aggregators'))

   identitiesInfo = IdentitiesInfo(infosInfo.getInfoPath('identities'))

   monitorsInfo = MonitorsInfo(infosInfo.getInfoPath('monitors'))
#
# Check that all enabled sites have an aggregator
#
   enabledSites = sitesInfo.getEnabledSites()
   for enabledSite in enabledSites:
      siteMembers = sitesInfo.getSiteMembers(enabledSite)
      if siteMembers:
         for siteMember in siteMembers:
            siteAggregator = aggregatorsInfo.getSiteAggregator(siteMember)
            if not siteAggregator:
               print("Enabled site %s has no aggregator" % (siteMember))
      else:
         siteAggregator = aggregatorsInfo.getSiteAggregator(enabledSite)
         if not siteAggregator:
            print("Enabled site %s has no aggregator" % (enabledSite))
#
# Check that all enabled tools destinations exist
#
   toolNames = toolsInfo.getToolNames()
   for toolName in toolNames:
      toolDestinations = toolsInfo.getToolDestinations(toolName)
      for toolDestination in toolDestinations:
         if not sitesInfo.siteExists(toolDestination):
            print("The destination %s listed for tool %s does not exist" % (toolDestination,toolName))
#
# Check that site managers exist
#
   expandedSiteNames = sitesInfo.getExpandedSiteNames(enabledSites)
   for expandedSiteName in expandedSiteNames:
      remoteManager = sitesInfo.getSiteKeyValue(expandedSiteName,'remoteManager')
      if remoteManager:
         if not remoteManager in usedRemoteManagers:
            usedRemoteManagers.append(remoteManager)
         if not managersInfo.managerExists(remoteManager):
            print("The manager %s listed for site %s does not exist" % (remoteManager,expandedSiteName))
#
# Check that tool managers exist
#
   toolNames = toolsInfo.getToolNames()
   for toolName in toolNames:
      toolDestinations = toolsInfo.getToolDestinations(toolName)
      for toolDestination in toolDestinations:
         selectedToolInfo = toolsInfo.selectTool(toolName,[toolDestination])
         remoteManager = selectedToolInfo['remoteManager']
         if remoteManager:
            if not remoteManager in usedRemoteManagers:
               usedRemoteManagers.append(remoteManager)
            if not managersInfo.managerExists(remoteManager):
               print("The manager %s listed for tool %s at site %s does not exist" % (remoteManager,toolName,toolDestination))
#
# Check for unused managers
#
   managerNames = managersInfo.getManagerNames()
   for managerName in managerNames:
      if not managerName in usedRemoteManagers:
         print("The manager %s is not used" % (managerName))
#
# Check that tool toolFiles exist
#
   toolNames = toolsInfo.getToolNames()
   for toolName in toolNames:
      toolDestinations = toolsInfo.getToolDestinations(toolName)
      for toolDestination in toolDestinations:
         selectedToolInfo = toolsInfo.selectTool(toolName,[toolDestination])
         toolFiles = selectedToolInfo['toolFiles']
         if toolFiles:
            if not toolFilesInfo.toolFileExists(toolFiles):
               print("The toolFiles %s listed for tool %s at site %s does not exist" % (toolFiles,toolName,toolDestination))
#
# Check that toolFiles dockerImage exist
#
   toolFileNames = toolFilesInfo.getToolFileNames()
   for toolFileName in toolFileNames:
      toolFileInfo = toolFilesInfo.getToolFilesInfo(toolFileName)
      if 'dockerImage' in toolFileInfo:
         if not dockerImagesInfo.dockerImageExists(toolFileInfo['dockerImage']):
            print("The dockerImage %s listed for toolFiles %s does not exist" % (toolFileInfo['dockerImage'],toolFileName))
#
# Check that all sites have an identity manager
#
   identityNames = identitiesInfo.getIdentityNames()
   expandedSiteNames = sitesInfo.getExpandedSiteNames(enabledSites)
   for expandedSiteName in expandedSiteNames:
      identityManagers = sitesInfo.getSiteKeyValue(expandedSiteName,'identityManagers')
      if not identityManagers:
         print("Enabled site %s has no identity manager" % (expandedSiteName))
      else:
         for identityManager in identityManagers:
            if not identityManager in identityNames:
               print("The identity manager %s listed for site %s does not exist" % (identityManager,expandedSiteName))
#
# Check that all TAPIS sites have an existing monitor
#
   enabledTapis2SiteNames = tapis2SitesInfo.getEnabledTapisSites()
   for enabledTapis2SiteName in enabledTapis2SiteNames:
      siteMonitorDesignator = tapis2SitesInfo.getSiteKeyValue(enabledTapis2SiteName,'siteMonitorDesignator')
      if not siteMonitorDesignator:
         print("Enabled tapis2site %s has no monitor" % (enabledTapis2SiteName))

   enabledTapis3SiteNames = tapis3SitesInfo.getEnabledTapisSites()
   for enabledTapis3SiteName in enabledTapis3SiteNames:
      siteMonitorDesignator = tapis3SitesInfo.getSiteKeyValue(enabledTapis3SiteName,'siteMonitorDesignator')
      if not siteMonitorDesignator:
         print("Enabled tapis3site %s has no monitor" % (enabledTapis3SiteName))
#
# Check that all sites have an existing monitor
#
   usedTapis2SiteMonitors = []
   usedTapis3SiteMonitors = []
   usedSiteMonitors = []
   expandedSiteNames = sitesInfo.getExpandedSiteNames(enabledSites)
   for expandedSiteName in expandedSiteNames:
      remoteBatchSystem = sitesInfo.getSiteKeyValue(expandedSiteName,'remoteBatchSystem')
      if remoteBatchSystem != "SCRIPT":
         siteMonitorDesignator = sitesInfo.getSiteKeyValue(expandedSiteName,'siteMonitorDesignator')
         if not siteMonitorDesignator:
            venueMechanism = sitesInfo.getSiteKeyValue(expandedSiteName,'venueMechanism')
            tapisSite = sitesInfo.getSiteKeyValue(expandedSiteName,'tapisSite')
            if   venueMechanism == 'tapis2' and tapisSite in enabledTapis2SiteNames:
               if not tapisSite in usedTapis2SiteMonitors:
                  usedTapis2SiteMonitors.append(tapisSite)
            elif venueMechanism == 'tapis2':
               print("Enabled site %s has no tapissite" % (expandedSiteName))
            elif venueMechanism == 'tapis3' and tapisSite in enabledTapis3SiteNames:
               if not tapisSite in usedTapis3SiteMonitors:
                  usedTapis3SiteMonitors.append(tapisSite)
            elif venueMechanism == 'tapis3':
               print("Enabled site %s has no tapissite" % (expandedSiteName))
            else:
               print("Enabled site %s has no monitor" % (expandedSiteName))
         else:
            if not siteMonitorDesignator in usedSiteMonitors:
               usedSiteMonitors.append(siteMonitorDesignator)
#
# Check for unused tapis sites
#
   for tapisSite in enabledTapis2SiteNames:
      if not tapisSite in usedTapis2SiteMonitors:
         print("The tapis2site %s is not used" % (tapisSite))

   for tapisSite in enabledTapis3SiteNames:
      if not tapisSite in usedTapis3SiteMonitors:
         print("The tapis3site %s is not used" % (tapisSite))
#
# Check that all used monitors have an identity manager
#
   for usedSiteMonitor in usedSiteMonitors:
      monitorInfo = monitorsInfo.getMonitorInfo(usedSiteMonitor)
      if 'identityManager' in monitorInfo:
         identityManager = monitorInfo['identityManager']
      else:
         identityManager = None
      if not identityManager:
         print("Monitor %s has no identity manager" % (usedSiteMonitor))
      else:
         if not identityManager in identityNames:
            print("The identity manager %s listed for monitor %s does not exist" % (identityManager,usedSiteMonitor))


