# @package      hubzero-submit-distributor
# @file         JobStatus.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
__version__ = '3.10.1'

import sys
import os
import re
import signal
import subprocess
import logging
from errno import EPIPE

from hubzero.submit.CommandParser    import CommandParser
from hubzero.submit.DaemonsInfo      import DaemonsInfo
from hubzero.submit.RemoteJobMonitor import RemoteJobMonitor

class JobStatus:
   def __init__(self,
                daemonsDirectory,
                daemonsConfigurationFile):
      self.logger           = logging.getLogger(__name__)
      self.version          = __version__
      self.operationMode    = 0

      configFilePath        = os.path.join(daemonsDirectory,daemonsConfigurationFile)
      daemonsInfo           = DaemonsInfo(configFilePath)
      listenURI             = daemonsInfo.getDaemonListenURI('jobMonitor','tcp')
      self.remoteJobMonitor = RemoteJobMonitor(listenURI)

      self.commandParser        = None
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'0'))
      self.statusIds            = []

      self.hubUserId = os.getuid()

      self.abortAttempted = False
      self.abortSignal    = 0

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError:
         pass


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError:
         pass


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODERUNSTATUS:
         self.statusIds = [int(statusId) for statusId in self.commandParser.getOption('statusIds')]
      else:
         exitCode = 1

      return(exitCode)

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      if not self.abortAttempted:
         self.abortAttempted = True
         self.abortSignal    = signalNumber


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.sigGEN_handler(signalNumber,frame)


   @staticmethod
   def __natsort(l):
      def tryfloat(s):
         try:
            return int(s)
         except:
            return s.lower()

      try:
         isinstance("", basestring)
         def isString(s):
            return isinstance(s,basestring)
      except NameError:
         def isString(s):
            return isinstance(s,str)

      def alphanum_key(s):
         """ Turn a string into a list of string and number chunks.
             "z23a" -> ["z", 23, "a"]
         """
         if isString(s):
            return [ tryfloat(c) for c in re.split('([0-9.]+)', s) ]
         else:
            return [ tryfloat(s) ]

      """ Sort the given list in the way that humans expect.
      """
      l.sort(key=alphanum_key)


   def report(self):
      """Report job status"""

      reportTime,reportedJobs = self.remoteJobMonitor.queryUserActiveJobStatus(self.hubUserId)
      if len(reportedJobs) > 0:
         reportAllIds = len(self.statusIds) == 0
         maximumRunNameLength = len('RunName')
         localJobIds = reportedJobs.keys()
         for localJobId in localJobIds:
            if reportAllIds or (localJobId in self.statusIds):
               instanceIds = reportedJobs[localJobId].keys()
               for instanceId in instanceIds:
                  runName,jobQueue,jobSite,jobStatus,jobStage = reportedJobs[localJobId][instanceId]
                  maximumRunNameLength = max(maximumRunNameLength,len(runName))

         self.__writeToStdout("%-*s  %12s  %8s  %20s  %s\n" % (maximumRunNameLength,'RunName', \
                                                               'submitRun','Instance','Status','Location'))
         localJobIds = list(reportedJobs.keys())
         self.__natsort(localJobIds)
         for localJobId in localJobIds:
            if reportAllIds or (localJobId in self.statusIds):
               instanceIds = list(reportedJobs[localJobId].keys())
               self.__natsort(instanceIds)
               for instanceId in instanceIds:
                  runName,jobQueue,jobSite,jobStatus,jobStage = reportedJobs[localJobId][instanceId]
                  if   '@' in jobSite:
                     queueSite = jobSite
                  elif jobQueue != '?':
                     queueSite = jobQueue + '@' + jobSite
                  else:
                     queueSite = jobSite
                  self.__writeToStdout("%-*s  %12s  %8s  %20s  %s\n" % (maximumRunNameLength,runName, \
                                                                        localJobId,instanceId,jobStatus,queueSite))

      del reportedJobs


